/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla.h"
#include "misc_string.h"
#include "GaleonWrapper.h"
#include "GlobalHistory.h"
#include "ProgressListener2.h"
#include "PrintProgressListener.h"

#include <gtkmozembed_internal.h>
#include <unistd.h>

#include "nsIContentViewer.h"
#include "nsIPermissionManager.h"
#include "nsIGlobalHistory.h"
#include "nsIDocShellHistory.h"
#include "nsIWebBrowserFind.h"
#include "nsIWebBrowserFocus.h"
#include "nsIDocument.h"
#include "nsISHEntry.h"
#include "nsISHistoryInternal.h"
#include "nsIHistoryEntry.h"
#include "nsIWebBrowserPrint.h"
#include "nsIURI.h"
#include "nsIPresShell.h"
#include "nsIMarkupDocumentViewer.h"
#include "nsIComponentManager.h"
#include "nsIDOMElement.h"
#include "nsIDOMNodeList.h"
#include "nsIScriptGlobalObject.h"
#include "nsIScriptContext.h"

#include "nsIDOMWindowInternal.h"
#include "nsICharsetConverterManager.h"
#include "nsICharsetConverterManager2.h"
#include "nsIInterfaceRequestor.h"
#include "nsIFocusController.h"
#include "nsIWebBrowserPersist.h"
#include "nsCWebBrowserPersist.h"
#include "nsNetUtil.h"
#include "nsIChromeEventHandler.h"
#include "nsIClipboardCommands.h"
#include "nsIDOMDocumentStyle.h"
#include "nsIDocShellTreeItem.h"
#include "nsIDocShellTreeNode.h"
#include "nsIDocShellTreeOwner.h"
#include "nsIHTMLContentContainer.h"
#include "nsICSSLoader.h"
#include "nsICSSStyleSheet.h"
#include "nsICSSLoaderObserver.h"
#include "nsIStyleSet.h"
#include "nsIDocumentObserver.h"
#include "nsCWebBrowser.h"
#include "nsReadableUtils.h"
#include "nsIDOMNSHTMLDocument.h"
#include "nsIDOMHTMLDocument.h"
#include "nsIDOMHTMLCollection.h"
#include "nsIDOMHTMLElement.h"
#include "nsIDOMHTMLImageElement.h"
#include "nsIDOMHTMLFormElement.h"
#include "nsIDOMHTMLAnchorElement.h"
#include "caps/nsIPrincipal.h"
#include "nsIDeviceContext.h"
#include "nsIPresContext.h"

GaleonWrapper::GaleonWrapper ()
{
}

GaleonWrapper::~GaleonWrapper ()
{
}

nsresult GaleonWrapper::Init (GaleonEmbed *galeon_embed)
{
	nsresult result;

	return_val_if_not_embed (galeon_embed, NS_ERROR_FAILURE);

	embed = galeon_embed;
	mGtkMozEmbed = GTK_MOZ_EMBED(embed->mozembed);

	gtk_moz_embed_get_nsIWebBrowser (mGtkMozEmbed,
					 getter_AddRefs(mWebBrowser));
	if (!mWebBrowser) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShellHistory> dsHistory = do_QueryInterface (DocShell);
	if (!dsHistory) return NS_ERROR_FAILURE;

	static NS_DEFINE_CID(kGlobalHistoryCID, GALEON_GLOBALHISTORY_CID);

	nsCOMPtr<nsIFactory> GHFactory;
	result = NS_NewGlobalHistoryFactory(getter_AddRefs(GHFactory));
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	result = nsComponentManager::RegisterFactory(kGlobalHistoryCID,
						     "Global history",
						     NS_GLOBALHISTORY_CONTRACTID,
						     GHFactory,
						     PR_TRUE);

	nsCOMPtr<nsIGlobalHistory> inst =  
		do_GetService(NS_GLOBALHISTORY_CONTRACTID, &result);
	return dsHistory->SetGlobalHistory(inst);
}

nsresult GaleonWrapper::GetDocShell (nsIDocShell **aDocShell)
{
        nsCOMPtr<nsIDocShellTreeItem> browserAsItem;
        browserAsItem = do_QueryInterface(mWebBrowser);
        if (!browserAsItem) return NS_ERROR_FAILURE;

        // get the owner for that item
        nsCOMPtr<nsIDocShellTreeOwner> treeOwner;
        browserAsItem->GetTreeOwner(getter_AddRefs(treeOwner));
        if (!treeOwner) return NS_ERROR_FAILURE;

        // get the primary content shell as an item
        nsCOMPtr<nsIDocShellTreeItem> contentItem;
        treeOwner->GetPrimaryContentShell(getter_AddRefs(contentItem));
        if (!contentItem) return NS_ERROR_FAILURE;

        // QI that back to a docshell
        nsCOMPtr<nsIDocShell> DocShell;
        DocShell = do_QueryInterface(contentItem);
        if (!DocShell) return NS_ERROR_FAILURE;

        *aDocShell = DocShell.get();

        NS_IF_ADDREF(*aDocShell);
        
        return NS_OK;
}

nsresult GaleonWrapper::SetOffline(PRBool aOffline)
{
	nsresult rv;
	nsCOMPtr<nsIIOService> io = do_GetService(NS_IOSERVICE_CONTRACTID, &rv);
	if (NS_FAILED(rv))
		return NS_ERROR_FAILURE;

	return io->SetOffline(aOffline);
}

nsresult GaleonWrapper::Print (nsIPrintSettings *options, PRBool preview)
{
	nsresult result;

	nsCOMPtr<nsIWebBrowserPrint> print(do_GetInterface(mWebBrowser, &result));
	if (NS_FAILED(result) || !print) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = mWebBrowser->GetContentDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	if (!preview)
	{
		GPrintListener *listener = new GPrintListener();
#if MOZILLA_VERSION > VERSION3(0,9,8)
		result = print->Print (options, listener);
#else
		result = print->Print (DOMWindow, options, listener);
#endif
	}
	else
	{
		char *filename;

		filename = g_strconcat(g_get_tmp_dir(),"/galeon-print-XXXXXX", NULL);
		/* get a name for the temporary file */
		int res;
		res = mkstemp (filename);
		if (res == -1)
			return NS_ERROR_FAILURE;
		close (res);
		mozilla_delete_temp_file_on_exit (filename);

		PRUnichar *uc_filename = mozilla_locale_to_unicode( filename );
		options->SetToFileName ( uc_filename );
		g_free( uc_filename );
		options->SetPrintToFile (TRUE);

		GPrintListener *listener =
			new GPrintListener(filename);
#if MOZILLA_VERSION > VERSION3(0,9,8)
		result = print->Print (options, listener);
#else
		result = print->Print (DOMWindow, options, listener);
#endif
		g_free (filename);
	}

	return result;
}

nsresult GaleonWrapper::GetPrintSettings (nsIPrintSettings **options)
{
	nsresult result;
	nsCOMPtr<nsIWebBrowserPrint> print(do_GetInterface(mWebBrowser, &result));
	if (NS_FAILED(result) || !print) return NS_ERROR_FAILURE;
#if MOZILLA_VERSION > VERSION3(0,9,8)
	return print->GetGlobalPrintSettings(options);
#else
	return print->GetPrintSettings(options);
#endif
}

nsresult GaleonWrapper::GetSHistory (nsISHistory **aSHistory)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = ContentNav->GetSessionHistory (getter_AddRefs (SessionHistory));
	if (!SessionHistory) return NS_ERROR_FAILURE;

	*aSHistory = SessionHistory.get();
	NS_IF_ADDREF (*aSHistory);

	return NS_OK;
}

nsresult GaleonWrapper::Destroy ()
{
      	mWebBrowser = nsnull;
	mChromeNav = nsnull;
	
	return NS_OK;
}

nsresult GaleonWrapper::GoToHistoryIndex (PRInt16 index)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	return  ContentNav->GotoIndex (index);
}

nsresult GaleonWrapper::SetZoom (float aZoom, PRBool reflow)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	if (reflow)
	{
		nsCOMPtr<nsIContentViewer> contentViewer;	
		result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
		if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

		nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
		if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

		return mdv->SetTextZoom (aZoom);
	}
	else
	{
		SetZoomOnDocshell (aZoom, DocShell);

		nsCOMPtr<nsIDocShellTreeNode> docShellNode(do_QueryInterface(DocShell));
		if (docShellNode)
		{
			PRInt32 i;
			PRInt32 n;
			docShellNode->GetChildCount(&n);
			for (i=0; i < n; i++) 
			{
				nsCOMPtr<nsIDocShellTreeItem> child;
				docShellNode->GetChildAt(i, getter_AddRefs(child));
				nsCOMPtr<nsIDocShell> childAsShell(do_QueryInterface(child));
				if (childAsShell) 
				{
					return SetZoomOnDocshell (aZoom, childAsShell);
				}
			}
		}
	}

	return NS_OK;
}

nsresult GaleonWrapper::SetZoomOnDocshell (float aZoom, nsIDocShell *DocShell)
{
	nsresult result;

	nsCOMPtr<nsIPresContext> PresContext;
	result = DocShell->GetPresContext (getter_AddRefs(PresContext));
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;
					
	nsCOMPtr<nsIDeviceContext> DeviceContext;
	result = PresContext->GetDeviceContext (getter_AddRefs(DeviceContext));

	return DeviceContext->SetTextZoom (aZoom);
}

nsresult GaleonWrapper::GetZoom (float *aZoom)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	return mdv->GetTextZoom (aZoom);
}

nsresult GaleonWrapper::GetFocusedDOMWindow (nsIDOMWindow **aDOMWindow)
{
	nsresult rv;
	
	nsCOMPtr<nsIWebBrowserFocus> focus = do_GetInterface(mWebBrowser, &rv);
	if (NS_FAILED(rv) || !focus) return NS_ERROR_FAILURE;

	rv = focus->GetFocusedWindow (aDOMWindow);
	if (NS_FAILED(rv))
		rv = mWebBrowser->GetContentDOMWindow (aDOMWindow);
	return rv;
}

nsresult GaleonWrapper::GetDOMDocument (nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	/* Use the current target document */
	if (mTargetDocument)
	{
		*aDOMDocument = mTargetDocument.get();

		NS_IF_ADDREF(*aDOMDocument);

		return NS_OK;
	}

	/* Use the focused document */
	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = GetFocusedDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_SUCCEEDED(result) && DOMWindow)
	{
		return DOMWindow->GetDocument (aDOMDocument);
	}

	/* Use the main document */
	return GetMainDOMDocument (aDOMDocument);
}

nsresult GaleonWrapper::GetMainDOMDocument (nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	return contentViewer->GetDOMDocument (aDOMDocument);
}

nsresult GaleonWrapper::SaveURI (const char *url, char *filename, DownloadAction action,
				 gpointer info)
{
        nsresult rv;

	nsString s;
	s.AssignWithConversion(url);
	nsCOMPtr<nsIURI> linkURI;
	rv = NS_NewURI(getter_AddRefs(linkURI), s);
	if (NS_FAILED(rv) || !linkURI) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebBrowserPersist> persist = 
			do_CreateInstance(NS_WEBBROWSERPERSIST_CONTRACTID, &rv);
	if (NS_FAILED(rv) || !persist) return NS_ERROR_FAILURE;

	nsCOMPtr<nsILocalFile> file;
	NS_NewLocalFile(filename, PR_TRUE, getter_AddRefs(file)); 
	if (NS_FAILED(rv) || !file) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMWindow> parent;
	rv = mWebBrowser->GetContentDOMWindow (getter_AddRefs(parent));

	GProgressListener2 *aProgress = new GProgressListener2 ();
	aProgress->InitForPersist (persist, parent,
					       linkURI, file,
					       action, info);
	mProgress = aProgress;

	switch (action)
	{
	case ACTION_VIEWSOURCE:
		persist->SetPersistFlags (nsIWebBrowserPersist::PERSIST_FLAGS_FROM_CACHE);
		break;
	case ACTION_SET_PIXMAP:
		break;
	case ACTION_FAVICON_EDITOR:
	case ACTION_FAVICON:
		persist->SetPersistFlags (nsIWebBrowserPersist::PERSIST_FLAGS_BYPASS_CACHE);
		break;
	default:
		size_t len = strlen (filename);
		if((filename[len-1] == 'z' && filename[len-2] == 'g') ||
		   (filename[len-1] == 'Z' && filename[len-2] == 'G'))
			persist->SetPersistFlags (nsIWebBrowserPersist::PERSIST_FLAGS_NO_CONVERSION);
		else
			persist->SetPersistFlags (nsIWebBrowserPersist::PERSIST_FLAGS_NONE);
		break;
	}
	return persist->SaveURI (linkURI, nsnull, file);
}

nsresult GaleonWrapper::SaveDocument (char *filename, const char *datapath,
				      DownloadAction action, PRBool mainDoc)
{
	nsresult rv;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	if (mainDoc)
		rv = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	else
		rv = GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(rv) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebBrowserPersist> persist = 
			do_CreateInstance(NS_WEBBROWSERPERSIST_CONTRACTID, &rv);
	if (NS_FAILED(rv) || !persist) return NS_ERROR_FAILURE;

	nsCOMPtr<nsILocalFile> file;
	rv = NS_NewLocalFile(filename, PR_TRUE, getter_AddRefs(file)); 
	if (NS_FAILED(rv) || !file) return NS_ERROR_FAILURE;

	nsCOMPtr<nsILocalFile> path;
	if (datapath)
		NS_NewLocalFile(datapath, PR_TRUE, getter_AddRefs(path));
	else
		path = nsnull;

	if (action)
	{

		nsCOMPtr<nsIDocument> aDocument =
				do_QueryInterface (aDOMDocument, &rv);
		if (NS_FAILED(rv) || !aDocument) return NS_ERROR_FAILURE;

		nsCOMPtr<nsIURI> aURI;
		rv = aDocument->GetDocumentURL (getter_AddRefs(aURI));
		if (NS_FAILED(rv) || !aURI) return NS_ERROR_FAILURE;

		nsCOMPtr<nsIDOMWindow> parent;
		mWebBrowser->GetContentDOMWindow (getter_AddRefs(parent));

		GProgressListener2 *aProgress = new GProgressListener2 ();
		aProgress->InitForPersist (persist, parent,
					       aURI, file, action, NULL);
		mProgress = aProgress;
	}
	return persist->SaveDocument (aDOMDocument, file, path, nsnull, 0, 0);
}

nsresult GaleonWrapper::GetSHInfo (PRInt32 *count, PRInt32 *index)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	SessionHistory->GetCount (count);
	SessionHistory->GetIndex (index);	

	return NS_OK;
}

nsresult GaleonWrapper::GetSHTitleAtIndex (PRInt32 index, PRUnichar **title)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIHistoryEntry> he;
	result = SessionHistory->GetEntryAtIndex (index, PR_FALSE,
						  getter_AddRefs (he));
	if (!NS_SUCCEEDED(result) || (!he)) return NS_ERROR_FAILURE;

	result = he->GetTitle (title);
	if (!NS_SUCCEEDED(result) || (!title)) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::GetSHUrlAtIndex (PRInt32 index, nsCString &url)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIHistoryEntry> he;
	result = SessionHistory->GetEntryAtIndex (index, PR_FALSE,
						  getter_AddRefs (he));
	if (NS_FAILED(result) || (!he)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIURI> uri;
	result = he->GetURI (getter_AddRefs(uri));
	if (NS_FAILED(result) || (!uri)) return NS_ERROR_FAILURE;

	result = uri->GetSpec(url);
	if (NS_FAILED(result) || url.IsEmpty()) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::Find (const PRUnichar *search_string, 
			      PRBool matchcase, PRBool search_backwards,
			      PRBool search_wrap_around,
			      PRBool search_for_entire_word,
			      PRBool search_in_frames,
			      PRBool *didFind)
{
	nsCOMPtr<nsIWebBrowserFind> finder (do_GetInterface(mWebBrowser));

	finder->SetSearchString (search_string);
	finder->SetFindBackwards (search_backwards);
	finder->SetWrapFind (search_wrap_around);
	finder->SetEntireWord (search_for_entire_word);
	finder->SetMatchCase (matchcase);
	finder->SetSearchFrames (search_in_frames);
	return finder->FindNext(didFind);
}

nsresult GaleonWrapper::ReloadDocument ()
{
	nsresult result;

	nsCOMPtr<nsIWebBrowserFocus> focus(do_GetInterface(mWebBrowser));

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = focus->GetFocusedWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIScriptGlobalObject> scriptGlobal = do_QueryInterface(DOMWindow);
	if (!scriptGlobal) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> docshell;
	if (NS_FAILED(scriptGlobal->GetDocShell(getter_AddRefs(docshell))))
        return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn = do_QueryInterface (docshell, &result);
	if (!wn || !NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;

	result = wn->Reload (nsIWebNavigation::LOAD_FLAGS_BYPASS_CACHE | 
			     nsIWebNavigation::LOAD_FLAGS_BYPASS_PROXY);
	if (!NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::GetMainDocumentUrl (nsCString &url)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	result = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIURI> uri;
	doc->GetDocumentURL(getter_AddRefs(uri));

	return uri->GetSpec (url);
}

nsresult GaleonWrapper::GetDocumentUrl (nsCString &url)
{
        nsresult result;

        nsCOMPtr<nsIDOMDocument> aDOMDocument;

        result = GetDOMDocument (getter_AddRefs(aDOMDocument));
        if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

        nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
        if(!doc) return NS_ERROR_FAILURE;

        nsCOMPtr<nsIURI> uri;
        doc->GetDocumentURL(getter_AddRefs(uri));

        uri->GetSpec (url);

        return NS_OK;
}

nsresult GaleonWrapper::GetDocumentTitle (char **title)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	result = GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	const nsString* t;
	t = doc->GetDocumentTitle();

	*title = mozilla_unicode_to_locale (t->get());

	return NS_OK;
}

nsresult  GaleonWrapper::CopyHistoryTo (GaleonWrapper *dest)
{
	nsresult result;
	int count,index;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_src = do_QueryInterface (DocShell,
							       &result);
	if (!wn_src) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsISHistory> h_src;
	result = wn_src->GetSessionHistory (getter_AddRefs (h_src));
	if (!NS_SUCCEEDED(result) || (!h_src)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> destDocShell;
	result = dest->GetDocShell (getter_AddRefs(destDocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_dest = do_QueryInterface (destDocShell,
								&result);
	if (!wn_dest) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsISHistory> h_dest;
	result = wn_dest->GetSessionHistory (getter_AddRefs (h_dest));
	if (!NS_SUCCEEDED (result) || (!h_dest)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHistoryInternal> hi_dest = do_QueryInterface (h_dest);
	if (!hi_dest) return NS_ERROR_FAILURE;

	h_src->GetCount (&count);
	h_src->GetIndex (&index);

	if (count) {
		nsCOMPtr<nsIHistoryEntry> he;
		nsCOMPtr<nsISHEntry> she;

		for (PRInt32 i = 0; i < count; i++) {

			result = h_src->GetEntryAtIndex (i, PR_FALSE,
							 getter_AddRefs (he));
			if (!NS_SUCCEEDED(result) || (!he))
				return NS_ERROR_FAILURE;

			she = do_QueryInterface (he);
			if (!she) return NS_ERROR_FAILURE;

			result = hi_dest->AddEntry (she, PR_TRUE);
			if (!NS_SUCCEEDED(result) || (!she))
				return NS_ERROR_FAILURE;
		}

		result = wn_dest->GotoIndex(index);
		if (!NS_SUCCEEDED(result)) return NS_ERROR_FAILURE;
	}

	return NS_OK;
}


nsresult GaleonWrapper::SetSitePermission (gboolean block, PRInt32 type)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> DOMDocument;
	result = GetDOMDocument (getter_AddRefs(DOMDocument));
	if (NS_FAILED(result) || !DOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface (DOMDocument);
	if (!doc) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsIURI> uri;
	doc->GetDocumentURL(getter_AddRefs(uri));

	nsCString url;
	uri->GetSpec (url);
	
	nsCOMPtr<nsIPermissionManager> permissionManager =
			do_CreateInstance (NS_PERMISSIONMANAGER_CONTRACTID);
	
	result = permissionManager->Add (url.get(), block ? PR_TRUE : PR_FALSE, type);

	return result;
}

nsresult GaleonWrapper::ForceCharacterSet (char *charset) 
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	nsAutoString charset_str;
	charset_str.AssignWithConversion (charset);
	result = mdv->SetForceCharacterSet (ToNewUnicode(charset_str));

	return result;
}

nsresult GaleonWrapper::CanCutSelection(PRBool *result)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CanCutSelection (result);
}

nsresult GaleonWrapper::CanCopySelection(PRBool *result)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CanCopySelection (result);
}

nsresult GaleonWrapper::CanPaste(PRBool *result)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CanPaste (result);
}

nsresult GaleonWrapper::CutSelection(void)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CutSelection ();
}

nsresult GaleonWrapper::CopySelection(void)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CopySelection ();
}

nsresult GaleonWrapper::Paste(void)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->Paste ();
}

nsresult GaleonWrapper::GetFaviconURL (nsCString &favicon_url)
{
	nsresult result;
	PRUint32 links_count;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	result = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsString strname;
	strname.AssignWithConversion("LINK");

	nsCOMPtr<nsIDOMNodeList> aLinks;
	result = aDOMDocument->GetElementsByTagName (strname, 
						     getter_AddRefs (aLinks));
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	result = aLinks->GetLength (&links_count);
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	for (PRUint32 i = 0; i < links_count; i++)
	{
		nsCOMPtr<nsIDOMNode> aLink;
		result = aLinks->Item (i, getter_AddRefs (aLink));
		if (NS_FAILED (result)) return NS_ERROR_FAILURE;

		nsCOMPtr<nsIDOMElement> linkElement;
		linkElement = do_QueryInterface (aLink);
		if (!linkElement) return NS_ERROR_FAILURE;

		nsAutoString attr;
		attr.AssignWithConversion("rel");
		nsAutoString value;
		result = linkElement->GetAttribute (attr, value);
		if (NS_FAILED(result)) return NS_ERROR_FAILURE;

		if (value.EqualsWithConversion ("SHORTCUT ICON", PR_TRUE) ||
		    value.EqualsWithConversion ("ICON", PR_TRUE))
		{
			nsAutoString attr;
			attr.AssignWithConversion("href");
			nsAutoString value;
			result = linkElement->GetAttribute (attr, value);
			if (NS_FAILED (result) || value.IsEmpty())
				return NS_ERROR_FAILURE;
			
			nsCString link;
			link.AssignWithConversion(value);

			nsCOMPtr<nsIDocument> doc = 
				do_QueryInterface (aDOMDocument);
			if(!doc) return NS_ERROR_FAILURE;
			
			nsCOMPtr<nsIURI> uri;
			result = doc->GetDocumentURL(getter_AddRefs(uri));
			if (NS_FAILED (result)) return NS_ERROR_FAILURE;

			result = uri->Resolve (link, favicon_url);
			if (NS_FAILED (result)) return NS_ERROR_FAILURE;
			
			return NS_OK;
		}
	}

	return NS_ERROR_FAILURE;
}

nsresult GaleonWrapper::GetStyleSheets(nsIDOMStyleSheetList** list)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> doc;
	result = GetDOMDocument(getter_AddRefs(doc));
	if (NS_FAILED(result) || !doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMDocumentStyle> docstyle(do_QueryInterface(doc));
	if (!docstyle) return NS_ERROR_FAILURE;

	result = docstyle->GetStyleSheets(list);
	if (NS_FAILED(result) || !list) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::LoadOverrideStyleSheet(char *css,
					       nsIStyleSheet **return_sheet)
{
	nsresult result;
	PRBool completed;

	/* catch necessary stuff */
	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresShell> ps;
	result = DocShell->GetPresShell (getter_AddRefs(ps));
	if (NS_FAILED(result) || !ps) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc;
	ps->GetDocument(getter_AddRefs(doc));

	nsCOMPtr<nsIHTMLContentContainer> container = do_QueryInterface(doc);
	if(!container) return NS_ERROR_FAILURE;

	nsICSSLoader *loader;
	result = container->GetCSSLoader (loader);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	/* load sheet */
	nsString s;
	s.AssignWithConversion(css);
	
	nsCOMPtr<nsIURI> uri;
	result = NS_NewURI(getter_AddRefs(uri), s);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	nsICSSStyleSheet *sheet;
	result = loader->LoadAgentSheet(uri, sheet, completed, nsnull);
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	/* catch stylesheet stuff and apply by appending it as a override
	 * sheet and enabling stylesheets */
	nsCOMPtr<nsIStyleSheet> styleSheet = do_QueryInterface(sheet);
	if (!styleSheet) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsIStyleSet> styleSet;
	result = ps->GetStyleSet(getter_AddRefs(styleSet));
	if (NS_FAILED(result) || !styleSet) return NS_ERROR_FAILURE;

	/* Override - Agent */
	styleSet->AppendUserStyleSheet(styleSheet);
	styleSheet->SetOwningDocument(doc);

	/* notify mozilla that stylesheets have changed */
	styleSheet->SetEnabled(PR_FALSE);
	styleSheet->SetEnabled(PR_TRUE);
	/* This is Right[tm], but not working for some people (?!)
	doc->SetStyleSheetDisabledState(styleSheet, PR_FALSE);
	*/

	*return_sheet = (nsIStyleSheet*)styleSheet;
	
	return NS_OK;
}

nsresult GaleonWrapper::RemoveOverrideStyleSheet(nsIStyleSheet *remove)
{
	nsresult result;

	/* get all necessary stuff */
	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	result = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	/* catch presshell */
	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIPresShell> ps;
	result = DocShell->GetPresShell (getter_AddRefs(ps));
	if (NS_FAILED(result) || !ps) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIStyleSet> styleSet;
	result = ps->GetStyleSet(getter_AddRefs(styleSet));
	if (NS_FAILED(result) || !styleSet) return NS_ERROR_FAILURE;

	/* remove it */
	styleSet->RemoveOverrideStyleSheet(remove);
	remove->SetEnabled(PR_FALSE);

	return NS_OK;
}

nsresult GaleonWrapper::GetLinkInterfaceItems (GList **list)
{
	nsresult result;
	PRUint32 links_count;
	char *relstr;

	/* we accept these rel=.. elements, specified by the w3c */
	const gchar *rel_types[] = {
		"START", "NEXT", "PREV", "PREVIOUS", "CONTENTS", "TOC", "INDEX",
		"GLOSSARY", "COPYRIGHT", "CHAPTER",  "SECTION",
		"SUBSECTION", "APPENDIX", "HELP", "TOP", "SEARCH", "MADE",
		"BOOKMARK", "HOME",
		NULL /* terminator, must be last */
	};

	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	result = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	/* get list of link elements*/
	nsString strname;
	strname.AssignWithConversion("LINK");

	nsCOMPtr<nsIDOMNodeList> aLinks;
	result = aDOMDocument->GetElementsByTagName (strname, 
						     getter_AddRefs (aLinks));
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	result = aLinks->GetLength (&links_count);
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	for (PRUint32 i = 0; i < links_count; i++)
	{
		/* get to the link element */
		nsCOMPtr<nsIDOMNode> aLink;
		result = aLinks->Item (i, getter_AddRefs (aLink));
		if (NS_FAILED (result)) return NS_ERROR_FAILURE;

		nsCOMPtr<nsIDOMElement> linkElement;
		linkElement = do_QueryInterface (aLink);
		if (!linkElement) return NS_ERROR_FAILURE;

		/* get rel=.. element */
		nsAutoString attr;
		attr.AssignWithConversion("rel");
		nsAutoString value;
		linkElement->GetAttribute (attr, value);

		if (value.IsEmpty())
		{
			attr.AssignWithConversion("rev");
			linkElement->GetAttribute (attr, value);
			if (value.IsEmpty()) continue;
		}

		relstr = ToNewCString (value);
		g_strup (relstr);

		/* check for elements we want */
		for (gint j = 0; (rel_types[j] != NULL); j++)
		{
			if (strcmp (relstr, rel_types[j]) == 0)
			{
				/* found one! */
				LinkInterfaceItem *lti =
					g_new0 (LinkInterfaceItem, 1);

				/* fill in struct */
				lti->type = (LinkInterfaceItemType) j;

				/* get href=.. element */
				nsAutoString attr;
				attr.AssignWithConversion("href");
				nsAutoString value;
				linkElement->GetAttribute (attr, value);

				if (value.IsEmpty())
				{
					g_free (lti);
					continue;
				}

				nsCString link;
				link.AssignWithConversion(value);

				/* resolve uri */
				nsCOMPtr<nsIDocument> doc = 
					do_QueryInterface (aDOMDocument);
				if(!doc) return NS_ERROR_FAILURE;
			
				nsCOMPtr<nsIURI> uri;
				doc->GetDocumentURL(getter_AddRefs(uri));

				nsCString href;
				result = uri->Resolve (link, href);
				if (NS_FAILED (result)) return NS_ERROR_FAILURE;
				lti->href = g_strdup (href.get());
		
				/* append to list of items */
				*list = g_list_append (*list, lti);
		
				/* get optional title=... element */
				attr.AssignWithConversion("title");
				linkElement->GetAttribute (attr, value);
				if (value.IsEmpty()) continue;

				char *title = ToNewCString (value);
				lti->title = misc_string_strip_newline (title);
				nsMemory::Free (title);
			}
		}

		nsMemory::Free (relstr);
	}

	return NS_OK;
}

nsresult GaleonWrapper::GetRealURL (nsCString &ret)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIURI> uri;
	result = ContentNav->GetCurrentURI (getter_AddRefs(uri));
	if (!NS_SUCCEEDED(result) || (!uri)) return NS_ERROR_FAILURE;

	result = uri->GetSpec(ret);
	if (!NS_SUCCEEDED(result) || ret.IsEmpty()) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::SelectAll (void)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->SelectAll ();
}

nsresult GaleonWrapper::ScrollUp (void)
{
	nsresult result;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = GetFocusedDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	DOMWindow->ScrollByLines(-1);

	return NS_OK;
}

nsresult GaleonWrapper::ScrollDown (void)
{
	nsresult result;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = GetFocusedDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	DOMWindow->ScrollByLines(1);
	
	return NS_OK;
}

nsresult GaleonWrapper::ScrollLeft (void)
{
	nsresult result;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = GetFocusedDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	DOMWindow->ScrollBy(-16, 0);
	
	return NS_OK;
}

nsresult GaleonWrapper::ScrollRight (void)
{
	nsresult result;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = GetFocusedDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	DOMWindow->ScrollBy(16, 0);
	
	return NS_OK;
}

nsresult GaleonWrapper::GetLastModified (gchar **ret)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	result = GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMNSHTMLDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsAutoString value;
	doc->GetLastModified(value);
	char *tmp = ToNewCString (value);
	*ret = g_strdup (tmp);
	nsMemory::Free (tmp);

	return NS_OK;
}

nsresult GaleonWrapper::GetImages (GList **ret)
{
	nsresult result;
	GHashTable *hash = g_hash_table_new (g_str_hash, g_str_equal);

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	result = GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMHTMLDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMHTMLCollection> col;
	doc->GetImages(getter_AddRefs(col));

	PRUint32 count, i;
	col->GetLength(&count);
	for (i = 0; i < count; i++)
	{
		nsCOMPtr<nsIDOMNode> node;
		col->Item(i, getter_AddRefs(node));
		if (!node) continue;

		nsCOMPtr<nsIDOMHTMLElement> element;
		element = do_QueryInterface(node);
		if (!element) continue;
		
		nsCOMPtr<nsIDOMHTMLImageElement> img;
		img = do_QueryInterface(element);
		if (!img) continue;

		ImageListItem *item = g_new0 (ImageListItem, 1);
		
		nsAutoString tmp;
		result = img->GetSrc (tmp);
		if (NS_SUCCEEDED(result))
		{
			char *c;
			c = ToNewCString(tmp);
			if (g_hash_table_lookup (hash, c))
			{
				nsMemory::Free (c);
				g_free (item);
				continue;
			}
			item->url = g_strdup (c);
			nsMemory::Free (c);
			g_hash_table_insert (hash, item->url,
					     GINT_TO_POINTER (TRUE));
		}
		result = img->GetAlt (tmp);
		if (NS_SUCCEEDED(result))
		{
			char *c;
			c = ToNewCString(tmp);
			item->alt = misc_string_strip_newline (c);
			nsMemory::Free (c);
		}
		result = element->GetTitle (tmp);
		if (NS_SUCCEEDED(result))
		{
			char *c;
			c = ToNewCString(tmp);
			item->title = misc_string_strip_newline (c);
			nsMemory::Free (c);
		}
		result = img->GetWidth (&(item->width));
		result = img->GetHeight (&(item->height));

		*ret = g_list_append (*ret, item);
	}

	g_hash_table_destroy (hash);

	return NS_OK;
}

nsresult GaleonWrapper::GetForms (GList **ret)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	result = GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMHTMLDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMHTMLCollection> col;
	doc->GetForms(getter_AddRefs(col));

	PRUint32 count, i;
	col->GetLength(&count);
	for (i = 0; i < count; i++)
	{
		nsCOMPtr<nsIDOMNode> node;
		col->Item(i, getter_AddRefs(node));
		if (!node) continue;

		nsCOMPtr<nsIDOMHTMLElement> element;
		element = do_QueryInterface(node);
		if (!element) continue;
		
		nsCOMPtr<nsIDOMHTMLFormElement> form;
		form = do_QueryInterface(element);
		if (!form) continue;

		FormListItem *item = g_new0 (FormListItem, 1);
		
		nsAutoString tmp;
		result = form->GetAction (tmp);
		if (NS_SUCCEEDED(result))
		{
			nsCString c, s;
			c.AssignWithConversion(tmp);

			nsCOMPtr<nsIDocument> doc = 
				do_QueryInterface (aDOMDocument);
			if(!doc) return NS_ERROR_FAILURE;
			
			nsCOMPtr<nsIURI> uri;
			doc->GetDocumentURL(getter_AddRefs(uri));

			result = uri->Resolve (c, s);

			item->action = s.Length() ? g_strdup (s.get()) : g_strdup (c.get());
		}
		result = form->GetMethod (tmp);
		if (NS_SUCCEEDED(result))
		{
			char *c;
			c = ToNewCString(tmp);
			item->method = g_strdup (c);
			nsMemory::Free (c);
		}
		result = form->GetName (tmp);
		if (NS_SUCCEEDED(result))
		{
			char *c;
			c = ToNewCString(tmp);
			item->name = g_strdup (c);
			nsMemory::Free (c);
		}

		*ret = g_list_append (*ret, item);
	}

	return NS_OK;
}

nsresult GaleonWrapper::GetLinks (GList **ret)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	result = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	/* first, get a list of <link> elements */
	PRUint32 links_count;

	nsString strname;
	strname.AssignWithConversion("LINK");

	nsCOMPtr<nsIDOMNodeList> aLinks;
	result = aDOMDocument->GetElementsByTagName (strname, 
						     getter_AddRefs (aLinks));
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	result = aLinks->GetLength (&links_count);
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	for (PRUint32 i = 0; i < links_count; i++)
	{
		nsCOMPtr<nsIDOMNode> aLink;
		result = aLinks->Item (i, getter_AddRefs (aLink));
		if (NS_FAILED (result)) continue;

		nsCOMPtr<nsIDOMElement> linkElement;
		linkElement = do_QueryInterface (aLink);
		if (!linkElement) continue;

		nsAutoString attr;
		attr.AssignWithConversion("href");
		nsAutoString value;
		linkElement->GetAttribute (attr, value);
		if (value.IsEmpty()) continue;

		nsCString link;
		link.AssignWithConversion(value);

		if (link.IsEmpty()) continue;
			
		nsCOMPtr<nsIDocument> doc = 
			do_QueryInterface (aDOMDocument);
		if(!doc) continue;
		
		nsCOMPtr<nsIURI> uri;
		doc->GetDocumentURL(getter_AddRefs(uri));

		nsCString tmp;
		result = uri->Resolve (link, tmp);
		
		LinkListItem *i = g_new0 (LinkListItem, 1);

		if (!tmp.IsEmpty())
		{
			i->url = g_strdup (tmp.get());
		}
		else
		{
			i->url = g_strdup (link.get());
		}

		attr.AssignWithConversion("title");
		linkElement->GetAttribute (attr, value);
		if (!value.IsEmpty())
		{
			char *s = ToNewCString (value);
			i->title = misc_string_strip_newline (s);
			nsMemory::Free (s);
		}

		attr.AssignWithConversion("rel");
		linkElement->GetAttribute (attr, value);
		if (!value.IsEmpty())
		{
			char *s = ToNewCString (value);
			i->rel = g_strdup (s);
			g_strdown (i->rel);
			nsMemory::Free (s);
		}
		if (!i->rel || strlen (i->rel) == 0)
		{
			attr.AssignWithConversion("rev");
			linkElement->GetAttribute (attr, value);
			if (!value.IsEmpty())
			{
				char *s = ToNewCString (value);
				i->rel = g_strdup (s);
				g_strdown (i->rel);
				nsMemory::Free (s);
			}
		}
		
		*ret = g_list_append (*ret, i);
	}

	/* next, get a list of anchors */
	nsCOMPtr<nsIDOMHTMLDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMHTMLCollection> col;
	doc->GetLinks(getter_AddRefs(col));

	PRUint32 count, i;
	col->GetLength(&count);
	for (i = 0; i < count; i++)
	{
		nsCOMPtr<nsIDOMNode> node;
		col->Item(i, getter_AddRefs(node));
		if (!node) continue;

		nsCOMPtr<nsIDOMHTMLElement> element;
		element = do_QueryInterface(node);
		if (!element) continue;
		
		nsCOMPtr<nsIDOMHTMLAnchorElement> lnk;
		lnk = do_QueryInterface(element);
		if (!lnk) continue;

		LinkListItem *i = g_new0 (LinkListItem, 1);

		nsAutoString tmp;
		char *c;
		result = lnk->GetHref (tmp);
		if (NS_SUCCEEDED(result))
		{
			c = ToNewCString(tmp);
			i->url = g_strdup (c);
			nsMemory::Free (c);
		}

		result = lnk->GetRel (tmp);
		if (NS_SUCCEEDED(result))
		{
			c = ToNewCString(tmp);
			i->rel = g_strdup (c);
			g_strdown (i->rel);
			nsMemory::Free (c);
		}

		if (!i->rel || strlen (i->rel) == 0)
		{
			result = lnk->GetRev (tmp);
			if (NS_SUCCEEDED(result))
			{
				c = ToNewCString(tmp);
				i->rel = g_strdup (c);
				g_strdown (i->rel);
				nsMemory::Free (c);
			}
		}

		i->title = mozilla_get_link_text (node);
		if (i->title == NULL)
		{
			result = element->GetTitle (tmp);
			if (NS_SUCCEEDED(result))
			{
				c = ToNewCString(tmp);
				i->title = misc_string_strip_newline (c);
				nsMemory::Free (c);
			}
		}


		*ret = g_list_append (*ret, i);
	}

	return NS_OK;
}

nsresult GaleonWrapper::EvaluateJS (char *script)
{
	nsresult rv;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	rv = mWebBrowser->GetContentDOMWindow(getter_AddRefs(DOMWindow));

	nsCOMPtr<nsIScriptGlobalObject> globalObject;
	globalObject = do_QueryInterface (DOMWindow);
	if (!globalObject) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIScriptContext> context;
	rv = globalObject->GetContext(getter_AddRefs(context));
	if (NS_FAILED(rv) || !context) {
		return NS_ERROR_FAILURE;
	}

	context->SetProcessingScriptTag(PR_TRUE);

	PRBool isUndefined;
	nsString ret;
	nsAutoString aScript;
	aScript.AssignWithConversion (script);
	context->EvaluateString(aScript, nsnull, nsnull, nsnull,
				0, nsnull, 
				ret, &isUndefined);  

	context->SetProcessingScriptTag(PR_FALSE);

	return NS_OK;
}

nsresult GaleonWrapper::PushTargetDocument (nsIDOMDocument *domDoc)
{
	mTargetDocument = domDoc;

	return NS_OK;
}

nsresult GaleonWrapper::PopTargetDocument ()
{
	mTargetDocument = nsnull;

	return NS_OK;
}
