/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla_i18n.h"
#include "mozilla.h"
#include "GaleonWrapper.h"
#include "EventContext.h"
#include "misc_string.h"

#include <time.h>
#include <gdk/gdkkeysyms.h>
#include <gtkmozembed_internal.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include "nsVoidArray.h"
#include "nsIPresShell.h"
#include "nsIDOMStyleSheetList.h"
#include "nsIDOMStyleSheet.h"

#include "nsString.h"
#include "nsXPIDLString.h"
#include "nsIPrefService.h"
#include "nsICharsetConverterManager.h"
#include "nsICharsetConverterManager2.h"
#include "nsIUnicodeEncoder.h"
#include "nsIUnicodeDecoder.h"
#include "nsICacheService.h"
#include "nsGfxCIID.h"
#include "nsNetCID.h"
#include "nsISupportsPrimitives.h"
#include "nsIPassword.h"
#include "nsIPasswordManager.h" 
#include "nsICookie.h"
#include "nsICookieManager.h" 
#include "nsIPermissionManager.h" 
#include "nsIImgManager.h" 
#include "nsIPermission.h" 
#include "nsIDOMWindow.h"
#include "nsIEmbeddingSiteWindow.h"
#include "nsIWindowWatcher.h"
#include "nsIWebBrowserChrome.h"
#include "nsIExternalHelperAppService.h"
#include "nsCExternalHandlerService.h"
#include "nsILocalFile.h"
#include "nsIJVMManager.h"
#include "nsIJSConsoleService.h"
#include "nsIPlatformCharset.h"
#include "nsIDOMHTMLLinkElement.h"
#include "nsIProtocolProxyService.h"
#include "nsReadableUtils.h"
#include "ProgressListener2.h"
#include "nsNetUtil.h"
#include "nsIWebBrowserPersist.h"
#include "nsCWebBrowserPersist.h"
#include "nsCWebBrowser.h"
#include "nsITransportSecurityInfo.h"
#include "nsIWebProgressListener.h"
#include "nsGUIEvent.h"
#include "nsIPrintOptions.h"
#include "nsIPrintSettings.h"
#include "nsIConsoleService.h"
#include "JSConsoleListener.h"
#include "nsIDOMCharacterData.h"
#include "nsIURL.h"

#define UTF8_ENCODER_ID NS_UNICODEENCODER_CONTRACTID_BASE"UTF-8"  
#define UTF8_DECODER_ID NS_UNICODEDECODER_CONTRACTID_BASE"UTF-8"

static NS_DEFINE_CID(kPrintOptionsCID, NS_PRINTOPTIONS_CID);
static NS_DEFINE_CID(kJVMManagerCID, NS_JVMMANAGER_CID);
static NS_DEFINE_CID(kPlatformCharsetCID, NS_PLATFORMCHARSET_CID);

#define GALEON_WRAPPER(wrapper)            \
GaleonWrapper *wrapper;                    \
return_if_not_embed(embed);	           \
g_return_if_fail (embed->wrapper != NULL); \
wrapper = (GaleonWrapper *)embed->wrapper;
	
#define GALEON_WRAPPER_IF_FAIL(wrapper, fail)        \
GaleonWrapper *wrapper;                              \
return_val_if_not_embed (embed, fail);	             \
g_return_val_if_fail (embed->wrapper != NULL, fail); \
wrapper = (GaleonWrapper *)embed->wrapper;

/* local function prototypes */
static char  *convert_ns_string_to_c_string (const nsString & ns_string);
static gboolean stylesheet_is_alternate (nsIDOMStyleSheet *item);

extern "C" gboolean
mozilla_save_prefs (void)
{
	nsCOMPtr<nsIPrefService> prefService = 
				 do_GetService (NS_PREFSERVICE_CONTRACTID);
	g_return_val_if_fail (prefService != nsnull, FALSE);

	nsresult rv = prefService->SavePrefFile (nsnull);
	return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

/**
 * mozilla_preference_set: set a string mozilla preference
 */
extern "C" gboolean
mozilla_preference_set(const char *preference_name, const char *new_value)
{
	g_return_val_if_fail (preference_name != NULL, FALSE);
	g_return_val_if_fail (new_value != NULL, FALSE);
	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));

	if (pref)
	{
		nsresult rv = pref->SetCharPref (preference_name, new_value);            
		return NS_SUCCEEDED (rv) ? TRUE : FALSE;
	}

	return FALSE;
}

/**
 * mozilla_preference_set_boolean: set a boolean mozilla preference
 */
extern "C" gboolean
mozilla_preference_set_boolean (const char *preference_name,
				gboolean new_boolean_value)
{
	g_return_val_if_fail (preference_name != NULL, FALSE);
  
	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));
  
	if (pref)
	{
		nsresult rv = pref->SetBoolPref (preference_name,
				new_boolean_value ? PR_TRUE : PR_FALSE);
		return NS_SUCCEEDED (rv) ? TRUE : FALSE;
	}

	return FALSE;
}

/**
 * mozilla_preference_set_int: set an integer mozilla preference
 */
extern "C" gboolean
mozilla_preference_set_int (const char *preference_name, int new_int_value)
{
	g_return_val_if_fail (preference_name != NULL, FALSE);

	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));

	if (pref)
	{
		nsresult rv = pref->SetIntPref (preference_name, new_int_value);
		return NS_SUCCEEDED (rv) ? TRUE : FALSE;
	}

	return FALSE;
}

/**
 * mozilla_preference_get_boolean: get a boolean mozilla preference
 */
extern "C" gboolean
mozilla_preference_get_boolean (const char *preference_name,
				gboolean default_value)
{
	PRBool value;

	g_return_val_if_fail (preference_name != NULL, FALSE);

	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));

	if (pref)
	{
		nsresult result;
		
		result = pref->GetBoolPref (preference_name, &value);
		if (NS_FAILED (result)) return default_value;
	}

	return (value == PR_TRUE) ? TRUE : FALSE;
}

/**
 * mozilla_preference_get_int: get an integer mozilla preference
 */
extern "C" gint
mozilla_preference_get_int (const char *preference_name)
{
	int value = -1;

	g_return_val_if_fail (preference_name != NULL, FALSE);

	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));

	if (pref)
	{
		pref->GetIntPref (preference_name, &value);
	}

	return value;
}

/**
 * mozilla_preference_get: get a string mozilla preference
 */
extern "C" gchar *
mozilla_preference_get(const char *preference_name)
{
	gchar *value = NULL;
	gchar *result = NULL;

	g_return_val_if_fail (preference_name != NULL, FALSE);
	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));

	if (pref)
	{
		pref->GetCharPref (preference_name, &value);            
	}

	/* it's allocated by mozilla, so I could not g_free it */
	if (value)
	{
		result = g_strdup (value);
		nsMemory::Free (value);
	}

	return result;
}

/**
 * mozilla_preference_remove: remove a mozilla preference
 */
extern "C" gboolean
mozilla_preference_remove (const char *preference_name)
{
	g_return_val_if_fail (preference_name != NULL, FALSE);

	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));

	if (pref)
	{
		nsresult rv = pref->ClearUserPref (preference_name);
		return NS_SUCCEEDED (rv) ? TRUE : FALSE;
	}

	return FALSE;
}

/**
 * mozilla_wrapper_init: initialize the mozilla wrapper
 * @attach_listener: attach event listener. Currently used for
 * links drag and drop only.
 */
extern "C" gpointer 
mozilla_wrapper_init (GaleonEmbed *embed)
{
	nsresult result;
	GaleonWrapper *wrapper = new GaleonWrapper ();
	result = wrapper->Init (embed);
	if (NS_FAILED(result))
		g_warning ("Wrapper initialization failed");
	return wrapper;
}

/**
 * mozilla_wrapper_destroy: destroy the mozilla wrapper
 */
extern "C" gboolean
mozilla_wrapper_destroy (GaleonEmbed *embed)
{
	nsresult result;
	GALEON_WRAPPER_IF_FAIL (wrapper, FALSE)
	
	result = wrapper->Destroy();
	embed->wrapper = NULL;
	delete wrapper;
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" gboolean mozilla_find (GaleonEmbed *embed, 
				  WrapperSearchProperties *properties)
{
	PRUnichar *search_string;		
	PRBool didFind;	
	GALEON_WRAPPER_IF_FAIL (wrapper, FALSE)

	search_string = mozilla_locale_to_unicode (properties->search_string);
	wrapper->Find (search_string, properties->match_case, 
		       properties->backwards, properties->wrap,
		       properties->entire_word, properties->search_frames,
		       &didFind);
	g_free (search_string);

	return didFind;
}

extern "C" gboolean mozilla_save_url (GaleonEmbed *embed, const char *url, 
				      char *filename, DownloadAction action,
				      gpointer *info)
{
	nsresult result;

	if (embed)
	{
		GALEON_WRAPPER_IF_FAIL (wrapper, FALSE)
		result = wrapper->SaveURI (url, filename, action, info);
	}
	else
	{
		nsresult rv;

		nsString s;
		s.AssignWithConversion(url);
		nsCOMPtr<nsIURI> linkURI;
		rv = NS_NewURI(getter_AddRefs(linkURI), s);
		if (NS_FAILED(rv) || !linkURI) return NS_ERROR_FAILURE;

		nsCOMPtr<nsIWebBrowserPersist> persist = 
			do_CreateInstance(NS_WEBBROWSERPERSIST_CONTRACTID, &rv);
		if (NS_FAILED(rv) || !persist) return NS_ERROR_FAILURE;

		nsCOMPtr<nsILocalFile> file;
		NS_NewLocalFile(filename, PR_TRUE, getter_AddRefs(file)); 
		if (NS_FAILED(rv) || !file) return NS_ERROR_FAILURE;

		GProgressListener2 *aProgress = new GProgressListener2 ();
		aProgress->InitForPersist (persist, NULL,
					       linkURI, file,
					       action, info);

		size_t len = strlen(filename);
		if((filename[len-1] == 'z' && filename[len-2] == 'g') ||
		   (filename[len-1] == 'Z' && filename[len-2] == 'G'))
			persist->SetPersistFlags (nsIWebBrowserPersist::PERSIST_FLAGS_NO_CONVERSION);
		else
			persist->SetPersistFlags (nsIWebBrowserPersist::PERSIST_FLAGS_NONE);

		return persist->SaveURI(linkURI, nsnull, file);
	}

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" gboolean mozilla_save_document (GaleonEmbed *embed, char *filename,
					   const char *datapath, DownloadAction action,
					   gboolean mainDoc)
{
	nsresult rv;
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	rv = wrapper->SaveDocument (filename, datapath, action,
				    mainDoc ? TRUE : FALSE);

	return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

extern "C" gboolean mozilla_reload (GaleonEmbed *embed)
{
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	nsresult result = wrapper->ReloadDocument ();
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_set_zoom: Sets the zoom factor of a embed
 */
extern "C" gboolean
mozilla_set_zoom (GaleonEmbed *embed, float f, gboolean reflow)
{
	gfloat current_zoom;
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)

	nsresult result = wrapper->GetZoom (&current_zoom);
	if (NS_FAILED (result) || current_zoom == f)
		return FALSE;

	result = wrapper->SetZoom (f, reflow);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_zoom: Gets the zoom factor of a embed
 */
extern "C" gboolean
mozilla_get_zoom (GaleonEmbed *embed, float *f)
{
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	nsresult result = wrapper->GetZoom (f);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_print: print a document
 */
extern "C" gboolean 
mozilla_print (GaleonEmbed *embed, PrintInfo *info, gboolean preview)
{
	nsresult result = NS_OK;
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	const static int frame_types[] = {
		nsIPrintSettings::kFramesAsIs,
		nsIPrintSettings::kSelectedFrame,
		nsIPrintSettings::kEachFrameSep
	};

	nsCOMPtr<nsIPrintOptions> printService(do_GetService(kPrintOptionsCID));
	g_assert(printService);

	nsCOMPtr<nsIPrintSettings> options;
	result = wrapper->GetPrintSettings(getter_AddRefs(options));
	g_assert (NS_SUCCEEDED (result));

	switch (info->pages)
	{
	case 0:
		break;
	case 1:
		options->SetPrintRange (nsIPrintSettings::kRangeSpecifiedPageRange);
		options->SetStartPageRange (info->from_page);
		options->SetEndPageRange (info->to_page);
		break;
	case 2:
		options->SetPrintRange (nsIPrintSettings::kRangeSelection);
		break;
	}

	options->SetMarginTop (info->top_margin);
	options->SetMarginBottom (info->bottom_margin);
	options->SetMarginLeft (info->left_margin);
	options->SetMarginRight (info->right_margin);

	PRUnichar * tempstr;

	tempstr = mozilla_locale_to_unicode("PostScript/default");
	printService->SetPrinterName(tempstr);
	g_free(tempstr);

	tempstr = mozilla_locale_to_unicode(info->header_left_string);
	options->SetHeaderStrLeft(tempstr);
	g_free(tempstr);
	
	tempstr = mozilla_locale_to_unicode(info->header_center_string);
	options->SetHeaderStrCenter(tempstr);
	g_free(tempstr);

	tempstr = mozilla_locale_to_unicode(info->header_right_string);
	options->SetHeaderStrRight(tempstr);
	g_free(tempstr);

	tempstr	= mozilla_locale_to_unicode(info->footer_left_string);
	options->SetFooterStrLeft(tempstr);
	g_free(tempstr);

	tempstr	= mozilla_locale_to_unicode(info->footer_center_string);
	options->SetFooterStrCenter(tempstr);
	g_free(tempstr);

	tempstr = mozilla_locale_to_unicode(info->footer_right_string);
	options->SetFooterStrRight(tempstr);
	g_free(tempstr);

	tempstr = mozilla_locale_to_unicode(info->file);
	options->SetToFileName (tempstr);
	g_free(tempstr);

	options->SetPrintToFile (info->print_to_file);

	tempstr = mozilla_locale_to_unicode(info->printer);
	options->SetPrintCommand (tempstr);
	g_free(tempstr);

	options->SetPrintReversed (info->reversed);
	options->SetPaperSize (info->paper);
	options->SetPrintInColor (info->print_color);
	options->SetOrientation (info->orientation);
	options->SetPrintFrameType (frame_types[info->frame_type]);

	options->SetPrintSilent (PR_TRUE);

	result = wrapper->Print(options, preview);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_session_history_get_url: Gets the session history and  ..
 * GtkMozWrapper. On return,  ..
 */
extern "C" char *
mozilla_session_history_get_url (GaleonEmbed *embed, int index)
{
	GALEON_WRAPPER_IF_FAIL(wrapper, NULL)
	
	nsresult rv;
	nsCString url;

	rv = wrapper->GetSHUrlAtIndex(index, url);

	return (NS_SUCCEEDED (rv) && !url.IsEmpty()) ? g_strdup(url.get()) : NULL;
}

extern "C" char *
mozilla_session_history_get_url_relative (GaleonEmbed *embed, int offset)
{
	int count, index;
	GALEON_WRAPPER_IF_FAIL(wrapper, NULL)
	
	wrapper->GetSHInfo (&count, &index);
	return mozilla_session_history_get_url (embed, index+offset);
}

/**
 * mozilla_session_history: Gets the session history and current position of a 
 * GtkMozWrapper. On return, *titles will be an array of char * wich must be freed.
 */
extern "C" gboolean
mozilla_session_history_get_all_titles (GaleonEmbed *embed, char **titles[], 
					int *count, int *index)
{
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	if (wrapper->GetSHInfo (count, index) == NS_ERROR_FAILURE)
		return FALSE;

	char **t = g_new(char *, *count);

	for (PRInt32 i = 0; i < *count; i++) {

		nsresult result;
		nsXPIDLString unicodeTitle;

		result = wrapper->GetSHTitleAtIndex(i,
						    getter_Copies(unicodeTitle));

		if (NS_FAILED(result))
		{
			return NS_OK;
		}
		
		t[i] = mozilla_unicode_to_locale (unicodeTitle.get());
	}
	*titles = t;

	return TRUE;
}

/**
 * mozilla_session_history_go: Goes to the SHEntry at the given index
 */
extern "C" gboolean
mozilla_session_history_go (GaleonEmbed *embed, int index)
{
	nsresult result;
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	result = wrapper->GoToHistoryIndex (index);
	
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_main_document_url: Get URL of main document
 * embed: Parent widget of document
 */
extern "C" gchar * 
mozilla_get_main_document_url (GaleonEmbed *embed)
{
	GALEON_WRAPPER_IF_FAIL(wrapper, NULL)
	
	nsCString url;
	nsresult rv = wrapper->GetMainDocumentUrl (url);

	return (NS_SUCCEEDED (rv) && !url.IsEmpty()) ? g_strdup(url.get()) : NULL;
}

/**
 * mozilla_get_document_url: Get URL of focused document (frame)
 * embed: Parent widget of document
 */
extern "C" gchar * 
mozilla_get_document_url (GaleonEmbed *embed)
{
	GALEON_WRAPPER_IF_FAIL(wrapper, NULL)
	
	nsCString url;
	nsresult rv = wrapper->GetDocumentUrl (url);

	return (NS_SUCCEEDED (rv) && !url.IsEmpty()) ? g_strdup(url.get()) : NULL;
}

/**
 * mozilla_get_frame_title: Get title of focused document (frame)
 * embed: Parent widget of document
 */
extern "C" gchar * 
mozilla_get_frame_title (GaleonEmbed *embed)
{
	GALEON_WRAPPER_IF_FAIL(wrapper, NULL)
	
	gchar *ret = NULL;
	wrapper->GetDocumentTitle (&ret);
	return ret;
}

/**
 * mozilla_get_document_title: get the document title in a locale specific
 * NULL terminated C string, using the private UniCode interface.
 */
extern "C" gchar * 
mozilla_get_document_title (GaleonEmbed *embed, gchar **title_utf8)
{
	PRUnichar *unicode_title;
	gchar *title;

	/* get the title in unicode */
	unicode_title = gtk_moz_embed_get_title_unichar 
		(GTK_MOZ_EMBED(embed->mozembed));

	/* do get utf8 version too */
	if (title_utf8 != NULL)
	{
		*title_utf8 = mozilla_unicode_to_utf8 (unicode_title);
	}

	/* attempt conversion */
	title = mozilla_unicode_to_locale (unicode_title);

	/* free unicode version */
	g_free (unicode_title);

	/* return it */
	return title;
}

/**
 * mozilla_get_link_message: get the link message in a locale specific
 * NULL terminated C string, using the private UniCode interface.
 */
extern "C" gchar * 
mozilla_get_link_message (GaleonEmbed *embed, gchar **link_message_utf8)
{
	gchar *link_message;

	PRUnichar *unicode_link_message;

	/* get the link message in unicode */
	unicode_link_message = gtk_moz_embed_get_link_message_unichar 
		(GTK_MOZ_EMBED(embed->mozembed));

	/* do get utf8 version too */
	if (link_message_utf8 != NULL)
	{
		*link_message_utf8
		    = mozilla_unicode_to_utf8 (unicode_link_message);
	}

	/* attempt conversion */
	link_message = mozilla_unicode_to_locale (unicode_link_message);

	/* free unicode version */
	g_free (unicode_link_message);

	/* return it */
	return link_message;
}

/**
 * mozilla_get_js_status: get the JavaScript status in a locale specific
 * NULL terminated C string, using the private UniCode interface.
 */
extern "C" gchar * 
mozilla_get_js_status (GaleonEmbed *embed, gchar **js_status_utf8)
{
	gchar *js_status;

	PRUnichar *unicode_js_status;

	/* get the link message in unicode */
	unicode_js_status = gtk_moz_embed_get_js_status_unichar 
		(GTK_MOZ_EMBED(embed->mozembed));

	/* do get utf8 version too */
	if (js_status_utf8 != NULL)
	{
		*js_status_utf8
		    = mozilla_unicode_to_utf8 (unicode_js_status);
	}

	/* attempt conversion */
	js_status = mozilla_unicode_to_locale (unicode_js_status);

	/* free unicode version */
	g_free (unicode_js_status);

	/* return it */
	return js_status;
}

struct nsKeyConverter {
  int vkCode; /* Platform independent key code */
  int keysym; /* GDK keysym key code */
};

/* From mozilla/widget/src/gtk/nsGtkEventHandler.cpp */
struct nsKeyConverter nsKeycodes[] = {

	{ NS_VK_CANCEL,     GDK_Cancel },
  	{ NS_VK_BACK,       GDK_BackSpace },
	{ NS_VK_TAB,        GDK_Tab },
  	{ NS_VK_CLEAR,      GDK_Clear },
  	{ NS_VK_RETURN,     GDK_Return },
  	{ NS_VK_SHIFT,      GDK_Shift_L },
  	{ NS_VK_CONTROL,    GDK_Control_L },
  	{ NS_VK_ALT,        GDK_Alt_L },
  	{ NS_VK_PAUSE,      GDK_Pause },
  	{ NS_VK_CAPS_LOCK,  GDK_Caps_Lock },
  	{ NS_VK_ESCAPE,     GDK_Escape },
  	{ NS_VK_SPACE,      GDK_space },
  	{ NS_VK_PAGE_UP,    GDK_Page_Up },
  	{ NS_VK_PAGE_DOWN,  GDK_Page_Down },
  	{ NS_VK_END,        GDK_End },
	{ NS_VK_HOME,       GDK_Home },
  	{ NS_VK_LEFT,       GDK_Left },
  	{ NS_VK_UP,         GDK_Up },
  	{ NS_VK_RIGHT,      GDK_Right },
  	{ NS_VK_DOWN,       GDK_Down },
  	{ NS_VK_PRINTSCREEN, GDK_Print },
  	{ NS_VK_INSERT,     GDK_Insert },
  	{ NS_VK_DELETE,     GDK_Delete },
  	{ NS_VK_MULTIPLY,   GDK_KP_Multiply },
  	{ NS_VK_ADD,        GDK_KP_Add },
  	{ NS_VK_SEPARATOR,  GDK_KP_Separator },
  	{ NS_VK_DECIMAL,    GDK_KP_Decimal },
  	{ NS_VK_DIVIDE,     GDK_KP_Divide },

  	{ NS_VK_COMMA,      GDK_comma },
  	{ NS_VK_PERIOD,     GDK_period },
  	{ NS_VK_SLASH,      GDK_slash },
  	{ NS_VK_BACK_SLASH, GDK_backslash },
  	{ NS_VK_BACK_QUOTE, GDK_grave },
  	{ NS_VK_OPEN_BRACKET, GDK_bracketleft },
  	{ NS_VK_CLOSE_BRACKET, GDK_bracketright },
  	{ NS_VK_SEMICOLON, GDK_semicolon },
  	{ NS_VK_QUOTE, GDK_apostrophe },

  	{ NS_VK_SUBTRACT, GDK_minus },
  	{ NS_VK_EQUALS, GDK_equal },
  	{ NS_VK_QUOTE, GDK_quotedbl },
  	{ NS_VK_BACK_SLASH, GDK_backslash },
  	{ NS_VK_1, GDK_1 },
  	{ NS_VK_2, GDK_2 },
  	{ NS_VK_3, GDK_3 },
  	{ NS_VK_4, GDK_4 },
  	{ NS_VK_5, GDK_5 },
  	{ NS_VK_6, GDK_6 },
  	{ NS_VK_7, GDK_7 },
  	{ NS_VK_8, GDK_8 },
  	{ NS_VK_9, GDK_9 },
  	{ NS_VK_0, GDK_0 }
};

/* FIXME mozilla should not make distinction between
 * upper and lowercase letters but apparently it does */
#define NS_VK_a 97
#define NS_VK_z 122

/**
 * mozilla_get_key_event_info:
 */
extern "C" gboolean
mozilla_get_key_event_info (GaleonEmbed *embed, gpointer event, WrapperKeyEventInfo *info)
{
	nsresult result;
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	EventContext context;
	context.Init ((nsIDOMEvent *)event, wrapper);
	result = context.GetKeyEventInfo (info);

	if (NS_FAILED(result)) return FALSE;
	
	int length = sizeof(nsKeycodes) / sizeof(struct nsKeyConverter);

	info->gdk_keyval = info->key;
	
	/* numeric pad numbers */
	if (info->key >= NS_VK_NUMPAD0 && 
	    info->key <= NS_VK_NUMPAD9 &&
	    (info->modifier & KEY_CODE))
	{
		info->gdk_keyval = info->key - NS_VK_NUMPAD0 + GDK_KP_0;
	}
	else /* function keys */
  	if (info->key >= NS_VK_F1 && 
	    info->key <= NS_VK_F24 &&
	    (info->modifier & KEY_CODE))
	{
    		info->gdk_keyval = info->key - NS_VK_F1 + GDK_F1;
	}
	else /*keys */ 
	if (info->key >= NS_VK_A && 
	    info->key <= NS_VK_Z)
	{
    		info->gdk_keyval = info->key - NS_VK_A + GDK_a;
		info->modifier |= SHIFT_KEY;
	}
	else 
	if (info->key >= NS_VK_a && 
	    info->key <= NS_VK_z)
	{
    		info->gdk_keyval = info->key - NS_VK_a + GDK_a;
	}
	else /* numbers */
	if (info->key >= NS_VK_0 && 
	    info->key <= NS_VK_9)
	{
		info->gdk_keyval = info->key - NS_VK_0 + GDK_0;
	}
	else /* misc other things */
	{
		for (int i = 0; i < length; i++) {
	    	if (nsKeycodes[i].vkCode == info->key)
      			info->gdk_keyval = nsKeycodes[i].keysym;
		}	
	}
	
	return TRUE;
}

/**
 * 
 */
extern "C" gboolean
mozilla_get_mouse_event_info (GaleonEmbed *embed, gpointer event, 
			      WrapperMouseEventInfo *context)
{
	nsresult result;
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	g_return_val_if_fail (event, FALSE);
	
	EventContext event_context;
	event_context.Init ((nsIDOMEvent*)event, wrapper);
	result = event_context.GetMouseEventInfo (context);

	nsCOMPtr<nsIDOMDocument> domDoc;
	result = event_context.GetTargetDocument (getter_AddRefs(domDoc));
	if (NS_SUCCEEDED(result))
	{
		result = wrapper->PushTargetDocument (domDoc);
	}

	/* for some reason we get different numbers on PPC, this fixes
	 * that up... -- MattA */
	if (context->button == 65536)
	{
		context->button = 1;
	}
	else if (context->button == 131072)
	{
		context->button = 2;
	}

	return NS_SUCCEEDED(result) ? TRUE : FALSE;
}

extern "C" gboolean
mozilla_pop_target_document (GaleonEmbed *embed)
{
	GALEON_WRAPPER_IF_FAIL (wrapper, FALSE);

	return wrapper->PopTargetDocument ();
}

/**
 * mozilla_free_context_info_sub: free info contained in WrapperContextInfo 
 * struct.
 */
extern "C" void
mozilla_free_context_info_sub (WrapperContextInfo *ctx)
{
	g_free (ctx->linktext);
	g_free (ctx->linktitle);
	g_free (ctx->imgalt);
	g_free (ctx->imgtitle);

	/* created by mozilla: */
	if (ctx->link) nsMemory::Free (ctx->link);
	if (ctx->linklang) nsMemory::Free (ctx->linklang);
	if (ctx->linktarget) nsMemory::Free (ctx->linktarget);
	if (ctx->linktype) nsMemory::Free (ctx->linktype);
	if (ctx->linkrel) nsMemory::Free (ctx->linkrel);
	if (ctx->linkrev) nsMemory::Free (ctx->linkrev);
	if (ctx->img) nsMemory::Free (ctx->img);
	if (ctx->imglongdesc) nsMemory::Free (ctx->imglongdesc);
	if (ctx->bgimg) nsMemory::Free (ctx->bgimg);
}

/**
 * mozilla_list_cookies: get a list of all saved cookies
 */
extern "C" GList *
mozilla_list_cookies (void)
{
	GList *cookies = NULL;
	nsresult result;

	nsCOMPtr<nsICookieManager> cookieManager = 
			do_CreateInstance (NS_COOKIEMANAGER_CONTRACTID);
	nsCOMPtr<nsISimpleEnumerator> cookieEnumerator;
	result = 
	    cookieManager->GetEnumerator (getter_AddRefs(cookieEnumerator));
	g_return_val_if_fail (NS_SUCCEEDED(result), NULL);	
	PRBool enumResult;
	for (cookieEnumerator->HasMoreElements(&enumResult) ;
	     enumResult == PR_TRUE ;
	     cookieEnumerator->HasMoreElements(&enumResult))
	{
		Cookie *c;
	
		nsCOMPtr<nsICookie> nsCookie;
		result = cookieEnumerator->GetNext (getter_AddRefs(nsCookie));
		g_return_val_if_fail (NS_SUCCEEDED(result), NULL);

		c = g_new0 (Cookie, 1);

		nsXPIDLCString transfer;

		nsCookie->GetHost (getter_Copies(transfer));
		c->base.domain = g_strdup (transfer.get());
		nsCookie->GetName (getter_Copies(transfer));
		c->name = g_strdup (transfer.get());
		nsCookie->GetValue (getter_Copies(transfer));
		c->value = g_strdup (transfer.get());
		nsCookie->GetPath (getter_Copies(transfer));
		c->path = g_strdup (transfer.get());
		
		PRBool isSecure;
		nsCookie->GetIsSecure (&isSecure);
		if (isSecure == PR_TRUE) 
			c->secure = g_strdup (_("Yes"));
		else 
			c->secure = g_strdup (_("No"));

		PRUint64 dateTime;
		nsCookie->GetExpires (&dateTime);
		c->expire = g_strdup_printf ("%s",ctime((time_t*)&dateTime));
		
		cookies = g_list_prepend (cookies, c);
	}	
	cookies = g_list_reverse (cookies);
	return cookies;
}

/**
 * mozilla_get_permissions:
 * @type: Type of permissions to get ( cookie or image )
 */

extern "C" GList *
mozilla_get_permissions (int type)
{
	GList *permissions = NULL;
	nsresult result;
	nsCOMPtr<nsIPermissionManager> permissionManager = 
			do_CreateInstance (NS_PERMISSIONMANAGER_CONTRACTID);
	nsCOMPtr<nsISimpleEnumerator> permissionEnumerator;
	result = permissionManager->
			GetEnumerator (getter_AddRefs(permissionEnumerator));
	g_return_val_if_fail (NS_SUCCEEDED(result), NULL);
	PRBool enumResult;
	for (permissionEnumerator->HasMoreElements(&enumResult) ;
	     enumResult == PR_TRUE ;
	     permissionEnumerator->HasMoreElements(&enumResult))
	{
		nsCOMPtr<nsIPermission> nsPermission;
		result = permissionEnumerator->
				GetNext (getter_AddRefs(nsPermission));
		g_return_val_if_fail (NS_SUCCEEDED(result), NULL);		

		PRInt32 cType;
		nsPermission->GetType (&cType);
		if (cType == type)
		{
			BlockedHost *b = g_new0 (BlockedHost, 1);
			gchar *tmp = NULL;

			nsPermission->GetHost (&tmp);
			b->domain = g_strdup (tmp);
			nsMemory::Free (tmp);

			PRBool cap;
			nsPermission->GetCapability (&cap);
			if (cap == PR_TRUE) 
				b->type = g_strdup (_("Allowed"));
			else 
				b->type = g_strdup (_("Blocked"));

			permissions = g_list_prepend (permissions, b);
		}
	}

	permissions = g_list_reverse (permissions);
	return permissions;
};

/**
 * mozilla_set_url_permission: change permissions for a URL
 */
extern "C" void
mozilla_set_url_permission (const char *imgURL, gint type, gboolean allow)
{
	nsCOMPtr<nsIPermissionManager> permissionManager =
			do_CreateInstance (NS_PERMISSIONMANAGER_CONTRACTID);

	permissionManager->Add (imgURL, allow ? PR_TRUE : PR_FALSE, type);
}

/**
 * mozilla_set_permission:
 * @permit: whether to block or allow
 * @type: type of permission to set ( cookie or image )
 */
 
extern "C" void
mozilla_set_permission (GaleonEmbed *embed, gboolean permit, int type)
{
	GALEON_WRAPPER(wrapper)
	wrapper->SetSitePermission (permit ? PR_TRUE : PR_FALSE, type);
}

/**
 * mozilla_remove_cookies:
 * @gone: list of cookies which are to be removed
 */
extern "C" gboolean 
mozilla_remove_cookies (GList *gone)
{
	nsresult result;
	nsCOMPtr<nsICookieManager> cookieManager =
			do_CreateInstance (NS_COOKIEMANAGER_CONTRACTID);

	for (GList *cookies = g_list_first(gone) ; cookies!=NULL ; 
	     cookies = g_list_next(cookies))
	{
		Cookie *c = (Cookie *)cookies->data;

		result = cookieManager->Remove (c->base.domain,c->name,
					 	c->path,
						PR_FALSE);
		if (NS_FAILED(result)) return FALSE;
	};
	return TRUE;
}

/**
 * mozilla_remove_permissions:
 * @gone: list of cookie or image permissions which are to be removed
 * @type: type of permissions ( cookies or images )
 */
extern "C" gboolean 
mozilla_remove_permissions (GList *gone, int type)
{
	nsresult result;
	nsCOMPtr<nsIPermissionManager> permissionManager =
			do_CreateInstance (NS_PERMISSIONMANAGER_CONTRACTID);

	for (GList *permissions = g_list_first(gone) ; permissions != NULL;
	     permissions = g_list_next(permissions))
	{
		BlockedHost *b = (BlockedHost *)permissions->data;
		result = permissionManager->Remove (b->domain,type);
		if (NS_FAILED(result)) return FALSE;
	};
	return TRUE;
}

/**
 * mozilla_copy_session_history: copy the session history to another embed
 */
extern "C" gboolean
mozilla_copy_session_history (GaleonEmbed *embed, GaleonEmbed *dest)
{
	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (dest != NULL, FALSE);
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	GaleonWrapper *dest_wrapper = (GaleonWrapper *)dest->wrapper;
	g_return_val_if_fail (wrapper != NULL, FALSE);
	g_return_val_if_fail (dest_wrapper != NULL, FALSE);
	nsresult result = wrapper->CopyHistoryTo (dest_wrapper);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_charsets: get a list of charset supported by mozilla
 */
extern "C" void
mozilla_get_charsets (GHashTable **charsets, GList **sorted_charset_titles)
{
	nsresult rv;
	PRUint32 cscount;
	PRUint32 translated_cscount = get_translated_cscount ();
	char *charset_str, *charset_title_str;

	nsCOMPtr<nsIAtom> docCharsetAtom;
	nsCOMPtr<nsICharsetConverterManager2> ccm2 =
		do_GetService (NS_CHARSETCONVERTERMANAGER_CONTRACTID, &rv);
	if (!NS_SUCCEEDED(rv)) return;

	nsCOMPtr <nsISupportsArray> cs_list;
	rv = ccm2->GetDecoderList (getter_AddRefs(cs_list));
	if (!NS_SUCCEEDED(rv)) return;

	rv = cs_list->Count(&cscount);
	*charsets = g_hash_table_new (g_str_hash, g_str_equal);
	for (PRUint32 i = 0; i < cscount; i++)
	{
		nsCOMPtr<nsISupports> cssupports =
					(dont_AddRef)(cs_list->ElementAt(i));
		nsCOMPtr<nsIAtom> csatom ( do_QueryInterface(cssupports) );
		nsString charset_ns = nsnull, charset_title_ns = nsnull;

		/* charset name */
		rv = csatom->ToString(charset_ns);
		charset_str = convert_ns_string_to_c_string (charset_ns);
		if (charset_str == NULL || strlen (charset_str) == 0)
		{
			continue;
		}

		/* charset readable title */
		rv = ccm2->GetCharsetTitle2(csatom, &charset_title_ns);
		charset_title_str = convert_ns_string_to_c_string 
							(charset_title_ns);
		if (charset_title_str == NULL || 
		    strlen (charset_title_str) == 0)
		{
			g_free (charset_title_str);
			charset_title_str = g_strdup (charset_str);
		}
		
		for (PRUint32 j = 0; j < translated_cscount; j++)
		{
			if (g_strcasecmp (
				charset_str, 
				charset_trans_array[j].charset_name) == 0)
			{
				g_free (charset_title_str);
				charset_title_str = (char *) 
					_(charset_trans_array[j].charset_title);
				break;
			}
		}

		/* fill the hash and the sorted list */
		g_hash_table_insert (*charsets, charset_title_str, charset_str);
		*sorted_charset_titles = g_list_insert_sorted (
						(GList*)*sorted_charset_titles,
						(gpointer)charset_title_str,
						(GCompareFunc)g_strcasecmp); 
	}
}

/**
 * mozilla_force_character_set: force the embed to use the specified
 * character set
 */
extern "C" void
mozilla_force_character_set (GaleonEmbed *embed, char *force_character_set)
{
	GALEON_WRAPPER(wrapper)
	wrapper->ForceCharacterSet (force_character_set);
}

/**
 * mozilla_clear_cache: Clear the global memory/disk cache
 */
extern "C" void
mozilla_clear_cache (unsigned long cache)
{
	nsresult rv;

	nsCOMPtr<nsICacheService> CacheService =
			do_GetService (NS_CACHESERVICE_CONTRACTID, &rv);
	if (NS_SUCCEEDED(rv))
		CacheService->EvictEntries(cache);
}

/* This nonsense is needed to get the allocators right */
static char *
convert_ns_string_to_c_string (const nsString & ns_string)
{
	char *c_string;
	nsCString ns_c_string;

	ns_c_string.AssignWithConversion (ns_string);

	c_string = g_strdup (ns_c_string.get());

	return c_string;
}

extern "C" gboolean
mozilla_get_favicon_location (GaleonEmbed *embed, gchar **url)
{
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	nsCString transfer;
	nsresult rv = wrapper->GetFaviconURL (transfer);
	if(transfer.IsEmpty())
		*url = NULL;
	else
		*url = g_strdup (transfer.get());
	
	return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

/**
 * mozilla_list_passwords: get a list of all saved username/password pairs
 */
extern "C" GList *
mozilla_list_passwords (PasswordType type)
{
	GList *passwords = NULL;
	nsresult result = NS_ERROR_FAILURE;

	nsCOMPtr<nsIPasswordManager> passwordManager =
			do_CreateInstance (NS_PASSWORDMANAGER_CONTRACTID);
	nsCOMPtr<nsISimpleEnumerator> passwordEnumerator;
	if (type == PASSWORD_PASSWORD)
		result = passwordManager->GetEnumerator 
				(getter_AddRefs(passwordEnumerator));
	else if (type == PASSWORD_REJECT)
		result = passwordManager->GetRejectEnumerator 
				(getter_AddRefs(passwordEnumerator));
	g_return_val_if_fail (NS_SUCCEEDED(result), NULL);	

	PRBool enumResult;
	for (passwordEnumerator->HasMoreElements(&enumResult) ;
	     enumResult == PR_TRUE ;
	     passwordEnumerator->HasMoreElements(&enumResult))
	{
		nsCOMPtr<nsIPassword> nsPassword;
		result = passwordEnumerator->GetNext 
					(getter_AddRefs(nsPassword));
		g_return_val_if_fail (NS_SUCCEEDED(result), NULL);

		Password *p = g_new0 (Password, 1);

		nsXPIDLCString transfer;
		nsPassword->GetHost (getter_Copies(transfer));
		p->host = g_strdup (transfer.get());

		if (type == PASSWORD_PASSWORD)
		{
			nsXPIDLString unicodeName;
			nsPassword->GetUser (getter_Copies(unicodeName));
			p->username = mozilla_unicode_to_locale (unicodeName.get());
		}
		
		passwords = g_list_prepend (passwords, p);
	}	
	passwords = g_list_reverse (passwords);
	return passwords;
}

/**
 * mozilla_remove_passwords:
 * @gone: list of passwords/rejcts which are to be removed
 * @type: whether list is passwords or rejects
 */
extern "C" gboolean 
mozilla_remove_passwords (GList *gone, PasswordType type)
{
	nsresult result = NS_ERROR_FAILURE;
	nsCOMPtr<nsIPasswordManager> passwordManager =
			do_CreateInstance (NS_PASSWORDMANAGER_CONTRACTID);

	for (GList *passwords = g_list_first(gone) ; passwords!=NULL ; 
	     passwords = g_list_next(passwords))
	{
		Password *p = (Password *)passwords->data;
		if (type == PASSWORD_PASSWORD)
		{
			PRUnichar *unicodeName = mozilla_locale_to_unicode
							(p->username);
			result = passwordManager->RemoveUser (p->host,
							      unicodeName);
			g_free (unicodeName);
		}
		else if (type == PASSWORD_REJECT)
		{
			result = passwordManager->RemoveReject (p->host);
		};

		if (NS_FAILED(result)) return FALSE;
	};
	return TRUE;
}

/**
 * mozilla_unicode_to_utf8: Encodes unicode string to UTF-8
 * @uniStr: The unicode string to encode
 */
extern "C" gchar *
mozilla_unicode_to_utf8 (const PRUnichar *uniStr)
{
	PRInt32 sSize,dSize;
	nsresult result;

	const nsString str (uniStr);

	sSize = str.Length ();

	nsCOMPtr<nsIUnicodeEncoder> 
		unicodeEncoder = do_CreateInstance (UTF8_ENCODER_ID);

	/* GetMaxLength returns a worst case prediction for the size
	   of the returned char*. Using it ensures that Convert will
	   not run out of space */
	result = unicodeEncoder->GetMaxLength (uniStr, sSize, &dSize);
	gchar *utfStr = g_new0 (gchar, dSize + 1);

	/* Convert must be passed the size of unicode string and
	   the size of the char* buffer so that bad things(tm)
	   won't happen. No null termination here. */
	result = unicodeEncoder->Convert (uniStr, &sSize, utfStr, &dSize);

	/* Finish ensures that the encoder is left in a clean state
	   for it's next use. */
	result = unicodeEncoder->Finish (utfStr, &dSize);

	/* Normally a null would need to appended at this point but
	   cStr was initialised to zero, so there's no need */
	return utfStr;
}

/**
 * mozilla_unicode_to_locale: Encodes unicode string to something
 * valid for the current locale (which can then be used in GTK+ labels).
 * @uniStr: The unicode string to encode
 */
extern "C" gchar *
mozilla_unicode_to_locale (const PRUnichar *uniStr)
{
	PRInt32 sSize,dSize;
	gchar *output;
	nsAutoString platformCharset;
	nsresult rv;

	/* sanity */
	if (uniStr == NULL)
	{
		return NULL;
	}

	nsCOMPtr<nsIPlatformCharset> svc;
	nsCOMPtr<nsICharsetConverterManager> ccm;
	nsCOMPtr<nsIUnicodeEncoder> encoder;
	const nsString str (uniStr);

	svc = do_GetService(NS_PLATFORMCHARSET_CONTRACTID, &rv);
	if (NS_SUCCEEDED(rv))
	{
		rv = svc->GetCharset(kPlatformCharsetSel_Menu,
				     platformCharset);
	}
	if (NS_SUCCEEDED(rv))
	{
		ccm = do_GetService (NS_CHARSETCONVERTERMANAGER_CONTRACTID,
				     &rv);
	}
	if (NS_SUCCEEDED(rv))
	{
		rv = ccm->GetUnicodeEncoder(&platformCharset,
					    getter_AddRefs(encoder));
	}
	if (NS_SUCCEEDED(rv)) {
		sSize = str.Length ();
		encoder->GetMaxLength (str.get(), sSize, &dSize);
		if (dSize)
		{
			output = g_new0 (gchar, dSize + 1);
			encoder->Convert (str.get(), &sSize, output, &dSize);
			encoder->Finish (output, &dSize);
			encoder->Reset ();
			return output;
		}
	}
	/* return empty string, if something fail */
	output = g_new0 (gchar, 1);
	*output = '\0';
	return output;
}

/**
 * mozilla_unicode_to_page_locale: Encodes unicode string to something
 * valid for the locale of the html page it came from/is going to.
 * Useful for form submission etc.
 * @uniStr: The unicode string to encode
 */
extern "C" gchar *
mozilla_unicode_to_page_locale (const PRUnichar *uniStr,
				nsAutoString &platformCharset)
{
	PRInt32 sSize,dSize;
	gchar *output;
	nsresult rv;

	/* sanity */
	if (uniStr == NULL)
	{
		return NULL;
	}

	nsCOMPtr<nsICharsetConverterManager> ccm;
	nsCOMPtr<nsIUnicodeEncoder> encoder;
	const nsString str (uniStr);

	ccm = do_GetService (NS_CHARSETCONVERTERMANAGER_CONTRACTID,
			     &rv);
	if (NS_SUCCEEDED(rv))
	{
		rv = ccm->GetUnicodeEncoder(&platformCharset,
					    getter_AddRefs(encoder));
	}
	if (NS_SUCCEEDED(rv)) {
		sSize = str.Length ();
		encoder->GetMaxLength (str.get(), sSize, &dSize);
		if (dSize)
		{
			output = g_new0 (gchar, dSize + 1);
			encoder->Convert (str.get(), &sSize, output, &dSize);
			encoder->Finish (output, &dSize);
			encoder->Reset ();
			return output;
		}
	}
	/* return empty string, if something fail */
	output = g_new0 (gchar, 1);
	*output = '\0';
	return output;
}

/**
 * mozilla_utf8_to_unicode: Decodes UTF-8 string to unicode
 * @utfStr: The unicode string to encode
 */
extern "C" PRUnichar *
mozilla_utf8_to_unicode (const gchar *utfStr)
{
	PRInt32 sSize,dSize;
	nsresult result;
	
	for (sSize = 0; utfStr[sSize] != 0; sSize++);

	nsCOMPtr<nsIUnicodeDecoder> unicodeDecoder = 
					do_CreateInstance (UTF8_DECODER_ID);

	/* Unicode decoding is much the same as encoding as
	   described in mozilla_session_history comments, 
	   but there is no Finish function needed to complete
	   the process */
	result = unicodeDecoder->GetMaxLength (utfStr, sSize, &dSize);
	PRUnichar *uniStr = g_new0 (PRUnichar, dSize + 1);

	result = unicodeDecoder->Convert (utfStr, &sSize, uniStr, &dSize);

	return uniStr;
}

/**
 * mozilla_locale_to_unicode: Decodes a string encoded for the current
 * locale into unicode. Used for getting text entered in a GTK+ entry
 * into a form which mozilla can use.
 * @locStr: The unicode string to encode
 */
extern "C" PRUnichar *
mozilla_locale_to_unicode (const gchar *locStr)
{
	PRInt32 sSize,dSize;
	PRUnichar *uniStr;
	nsAutoString platformCharset;
	nsresult rv;

	/* sanity */
	if (locStr == NULL)
	{
		return NULL;
	}

	nsCOMPtr<nsIPlatformCharset> svc;
	nsCOMPtr<nsICharsetConverterManager> ccm;
	nsCOMPtr<nsIUnicodeDecoder> decoder;

	svc = do_GetService(NS_PLATFORMCHARSET_CONTRACTID, &rv);
	if (NS_SUCCEEDED(rv))
	{
		rv = svc->GetCharset(kPlatformCharsetSel_Menu,
				     platformCharset);
	}
	if (NS_SUCCEEDED(rv))
	{
		ccm = do_GetService (NS_CHARSETCONVERTERMANAGER_CONTRACTID,
				     &rv);
	}
	if (NS_SUCCEEDED(rv))
	{
		rv = ccm->GetUnicodeDecoder(&platformCharset,
					    getter_AddRefs(decoder));
	}
	if (NS_SUCCEEDED(rv))
	{
		/* FIXME: if locStr in UTF-16/UCS-2/UCS-4 etc,
			  this will not work */
		sSize = strlen(locStr);
		decoder->GetMaxLength (locStr, sSize, &dSize);
		if (dSize)
		{
			uniStr = g_new0 (PRUnichar, dSize + 1);
			decoder->Convert (locStr, &sSize, uniStr, &dSize);
			uniStr[dSize] = '\0';
			decoder->Reset ();
			return uniStr;
		}
	}
	/* return empty string, if something fail */
	uniStr = g_new0 (PRUnichar, 1);
	uniStr[0] = '\0';
	return uniStr;
}

/**
 * mozilla_page_locale_to_unicode: Decodes a string encoded for the current
 * page's locale into unicode. Useful for form submission etc.
 * @locStr: The unicode string to encode
 */
extern "C" PRUnichar *
mozilla_page_locale_to_unicode (const gchar *locStr,
				nsAutoString &platformCharset)
{
	PRInt32 sSize,dSize;
	PRUnichar *uniStr;
	nsresult rv;

	/* sanity */
	if (locStr == NULL)
	{
		return NULL;
	}

	nsCOMPtr<nsIPlatformCharset> svc;
	nsCOMPtr<nsICharsetConverterManager> ccm;
	nsCOMPtr<nsIUnicodeDecoder> decoder;

	svc = do_GetService(NS_PLATFORMCHARSET_CONTRACTID, &rv);
	if (NS_SUCCEEDED(rv))
	{
		rv = svc->GetCharset(kPlatformCharsetSel_Menu,
				     platformCharset);
	}
	if (NS_SUCCEEDED(rv))
	{
		ccm = do_GetService (NS_CHARSETCONVERTERMANAGER_CONTRACTID,
				     &rv);
	}
	if (NS_SUCCEEDED(rv))
	{
		rv = ccm->GetUnicodeDecoder(&platformCharset,
					    getter_AddRefs(decoder));
	}
	if (NS_SUCCEEDED(rv))
	{
		/* FIXME: if locStr in UTF-16/UCS-2/UCS-4 etc,
			  this will not work */
		sSize = strlen(locStr);
		decoder->GetMaxLength (locStr, sSize, &dSize);
		if (dSize)
		{
			uniStr = g_new0 (PRUnichar, dSize + 1);
			decoder->Convert (locStr, &sSize, uniStr, &dSize);
			uniStr[dSize] = '\0';
			decoder->Reset ();
			return uniStr;
		}
	}
	/* return empty string, if something fail */
	uniStr = g_new0 (PRUnichar, 1);
	uniStr[0] = '\0';
	return uniStr;
}

extern "C" gchar *
mozilla_utf8_to_locale (const gchar* utfstr)
{
	if (utfstr == NULL)
	{
		return NULL;
	}
	PRUnichar* unistr = mozilla_utf8_to_unicode(utfstr);
	if (unistr == NULL)
	{
		return NULL;
	}
	gchar* locstr = mozilla_unicode_to_locale(unistr);
	g_free(unistr);
	return locstr;
}

extern "C" gchar *
mozilla_locale_to_utf8 (const gchar* locstr)
{
	if (locstr == NULL)
	{
		return NULL;
	}
	PRUnichar* unistr = mozilla_locale_to_unicode(locstr);
	if (unistr == NULL)
	{
		return NULL;
	}
	gchar* utfstr = mozilla_unicode_to_utf8(unistr);
	g_free(unistr);
	return utfstr;
}

/**
 * mozilla_find_gtk_parent: retrives gtkwidget pointer to parent
 * of domWindow. It will be the gtkmozembed that holds it.
 * @domWindow: nsIDOMWindow to find gtk parent of.
 */
extern "C" GtkWidget *
mozilla_find_gtk_parent (nsIDOMWindow *domWindow)
{
	nsresult result;
	GaleonWindow *gwindow;
	
        nsCOMPtr<nsIWindowWatcher> wwatch 
		(do_GetService("@mozilla.org/embedcomp/window-watcher;1"));
        if (!wwatch) return nsnull;

        nsCOMPtr<nsIWebBrowserChrome> windowChrome;
        result = wwatch->GetChromeForWindow (domWindow,
        				     getter_AddRefs(windowChrome));
        if (NS_FAILED(result)) return nsnull;

        nsCOMPtr<nsIEmbeddingSiteWindow> window
        		(do_QueryInterface(windowChrome, &result));
        if (NS_FAILED(result)) return nsnull;

	GtkWidget *mozembed;
	result = window->GetSiteWindow ((void **)&mozembed);
	if (NS_FAILED(result)) return nsnull;

	GaleonEmbed *embed = (GaleonEmbed*)
		gtk_object_get_data (GTK_OBJECT (mozembed), "GaleonEmbed");
	g_assert (embed != NULL);
	g_assert (embed->magic == GALEON_EMBED_MAGIC);

	gwindow = embed->parent_window;
	/* The nautilus view has not a GaleonWindpw */
	if (gwindow == NULL) return NULL;
	
	return gwindow->wmain;
}

extern "C" gboolean
mozilla_set_offline (GaleonEmbed *embed, gboolean offline)
{
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	nsresult result = wrapper->SetOffline (offline);

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" gboolean
mozilla_can_cut_selection (GaleonEmbed *embed)
{
	gboolean result;
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	wrapper->CanCutSelection (&result);

	return result;
}

extern "C" gboolean
mozilla_can_copy_selection (GaleonEmbed *embed)
{
	gboolean result;
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)
	
	wrapper->CanCopySelection (&result);

	return result;
}

extern "C" gboolean
mozilla_can_paste (GaleonEmbed *embed)
{
	gboolean result;
	GALEON_WRAPPER_IF_FAIL(wrapper, FALSE)

	wrapper->CanPaste (&result);

	return result;
}

extern "C" void
mozilla_cut_selection (GaleonEmbed *embed)
{
	GALEON_WRAPPER(wrapper)
	wrapper->CutSelection ();
}

extern "C" void
mozilla_copy_selection (GaleonEmbed *embed)
{
	GALEON_WRAPPER(wrapper)
	wrapper->CopySelection ();
}

extern "C" void
mozilla_paste (GaleonEmbed *embed)
{
	GALEON_WRAPPER(wrapper)
	wrapper->Paste ();
}

extern "C" void
mozilla_delete_temp_file_on_exit (const char *filename)
{
	nsresult rv;
	nsCOMPtr<nsILocalFile> localFile;
	rv = NS_NewLocalFile (filename, PR_TRUE, getter_AddRefs(localFile));
	if (NS_FAILED(rv)) return;

	nsCOMPtr<nsIExternalHelperAppService> helperService =
			do_GetService (NS_EXTERNALHELPERAPPSERVICE_CONTRACTID);

	nsCOMPtr<nsPIExternalAppLauncher> appLauncher =
					do_QueryInterface (helperService, &rv);
	if (NS_SUCCEEDED(rv))
	{
		appLauncher->DeleteTemporaryFileOnExit (localFile);
	}
}

extern "C" gboolean
mozilla_show_java_console (void)
{
	nsresult rv;
	nsCOMPtr<nsIJVMManager> JVMManager = do_GetService (kJVMManagerCID,
							    &rv);
	if (NS_FAILED(rv)) return FALSE;

	rv = JVMManager->ShowJavaConsole ();

	return rv ? TRUE : FALSE;
}

extern "C" gboolean
mozilla_show_javascript_console (void)
{
	nsresult rv;
	nsCOMPtr<nsIJSConsoleService> JSConsoleService =
		do_GetService ("@mozilla.org/embedcomp/jsconsole-service;1",
			       &rv);
	if (NS_FAILED(rv)) return FALSE;

	rv = JSConsoleService->Open (nsnull);

	return rv ? TRUE : FALSE;
}

/* Argh, we need to support every possible combination of named and unnamed
 * stylesheets. */
extern "C" GList *
mozilla_get_alternate_stylesheets (GaleonEmbed *embed)
{
	gchar *defaultname = NULL;
	gint found = 0;
	GList *csslist = NULL;
	nsresult result;
	GALEON_WRAPPER_IF_FAIL(wrapper, NULL)
	
	nsCOMPtr<nsIDOMStyleSheetList> list;
	result = wrapper->GetStyleSheets(getter_AddRefs(list));
	if (NS_FAILED(result) || !list) return NULL;
	
	PRUint32 count, i; 
	result = list->GetLength(&count);
	if (NS_FAILED(result)) return NULL;

	for (i = 0; i < count; i++)
	{
		nsCOMPtr<nsIDOMStyleSheet> item;
		result = list->Item(i, getter_AddRefs(item));
		if (NS_FAILED(result) || !item) continue;

		found++;

		nsString string;
		result = item->GetTitle(string);
		if (NS_FAILED(result)) continue;

		if (string.IsEmpty()) continue;

		if (!stylesheet_is_alternate (item))
		{
			if (defaultname) g_free (defaultname);
			defaultname = convert_ns_string_to_c_string (string);
			continue;
		}

		AlternateStyleSheet *asheet = g_new0 (AlternateStyleSheet, 1);
		asheet->name = convert_ns_string_to_c_string(string);
		asheet->sheet = item;
		asheet->type = STYLESHEET_ALTERNATE;

		csslist = g_list_append (csslist, asheet);
	}

	if (found > 0)
	{
		AlternateStyleSheet *asheet = g_new0 (AlternateStyleSheet, 1);
		if (defaultname)
		{
			asheet->name = g_strdup (defaultname);
		}
		else
		{
			asheet->name = g_strdup (_("Default"));
		}
		asheet->sheet = NULL;
		asheet->type = STYLESHEET_DEFAULT;

		csslist = g_list_append (csslist, asheet);

		if (defaultname) g_free (defaultname);
	}

	if (found > 0)
	{
		/* prepend None item if any sheets were found */
		AlternateStyleSheet *asheet = g_new0 (AlternateStyleSheet, 1);
		asheet->name = g_strdup (_("None"));
		asheet->sheet = NULL;
		asheet->type = STYLESHEET_NONE;

		csslist = g_list_prepend (csslist, asheet);
	}

	return csslist;
}

extern "C" void
mozilla_set_alternate_stylesheet (GaleonEmbed *embed, AlternateStyleSheet *css)
{
	nsresult result;
	GALEON_WRAPPER(wrapper)
	
	nsCOMPtr<nsIDOMStyleSheetList> list;
	result = wrapper->GetStyleSheets(getter_AddRefs(list));
	if (NS_FAILED(result) || !list) return;
	
	PRUint32 count, i; 
	result = list->GetLength(&count);
	if (NS_FAILED(result)) return;

	for (i = 0; i < count; i++)
	{
		nsCOMPtr<nsIDOMStyleSheet> item;
		result = list->Item(i, getter_AddRefs(item));
		if (NS_FAILED(result) || !item) continue;
		
		nsString string;
		result = item->GetTitle(string);
		if (NS_FAILED(result)) continue;

		/*
		 * if none is requtested, disable all.
		 * if the sheet matches the requested sheet, enable
		 * if the default is requested, load all sheets except
		 *   alternates.
		 * if an alternate is requested, also nameless sheets are
		 *   loaded.
		 */
		if      (item == css->sheet)
		{
			item->SetDisabled(FALSE);
		}
		else if (css->type == STYLESHEET_DEFAULT &&
			 !stylesheet_is_alternate (item))
		{
			item->SetDisabled(FALSE);
		}
		else if (css->type == STYLESHEET_ALTERNATE && string.IsEmpty())
	  	{
			item->SetDisabled(FALSE);
		}
		else
		{
			item->SetDisabled(TRUE);
		}
	}

}

extern "C" AlternateStyleSheet*
mozilla_get_selected_stylesheet (GaleonEmbed *embed)
{
	gboolean found = FALSE;
	nsresult result;
	GALEON_WRAPPER_IF_FAIL(wrapper, NULL)
	
	nsCOMPtr<nsIDOMStyleSheetList> list;
	result = wrapper->GetStyleSheets(getter_AddRefs(list));
	if (NS_FAILED(result) || !list) return NULL;
	
	PRUint32 count, i; 
	result = list->GetLength(&count);
	if (NS_FAILED(result)) return NULL;

	for (i = 0; i < count; i++)
	{
		nsCOMPtr<nsIDOMStyleSheet> item;
		result = list->Item(i, getter_AddRefs(item));
		if (NS_FAILED(result) || !item) continue;

		nsString string;
		result = item->GetTitle(string);
		if (NS_FAILED(result)) continue;

		PRBool disabled;
		item->GetDisabled(&disabled);
		
		if (string.IsEmpty())
		{
			/* fine, but let's try to get something more sensible */
			if (disabled == PR_FALSE) found = TRUE;
			continue;
		}
		
		if (disabled == PR_FALSE)
		{
			AlternateStyleSheet *asheet =
				g_new0 (AlternateStyleSheet, 1);
			asheet->name = convert_ns_string_to_c_string(string);
			asheet->sheet = item;
			asheet->type = STYLESHEET_ALTERNATE;
			return asheet;
		}
	}

	if (found)
	{
		AlternateStyleSheet *asheet = g_new0 (AlternateStyleSheet, 1);
		return asheet;
	}
	else
		return NULL;
}

static gboolean
stylesheet_is_alternate (nsIDOMStyleSheet *item)
{
	nsresult result;
	gboolean ret = FALSE;

	nsCOMPtr<nsIDOMNode> node;
	result = item->GetOwnerNode(getter_AddRefs(node));
	if (NS_FAILED(result)) return FALSE;

	nsCOMPtr<nsIDOMHTMLLinkElement> link = do_QueryInterface(node);
	
	if (link)
	{
		nsString str;
		link->GetRel(str);
		gchar *tmp = convert_ns_string_to_c_string(str);
		
		g_strdown (tmp);
		
		if (strstr (tmp, "alternate") != NULL) ret = TRUE;

		g_free (tmp);
	}

	return ret;
}

extern "C" gpointer
mozilla_set_user_sheet (GaleonEmbed *embed, gchar *sheet)
{
	nsIStyleSheet *ret = nsnull;
	GALEON_WRAPPER_IF_FAIL(wrapper, NULL)

	gchar *tmp = g_strconcat ("file://", sheet, NULL);
	wrapper->LoadOverrideStyleSheet(tmp, &ret);
	g_free (tmp);

	return (gpointer*)ret;
}

extern "C" void
mozilla_remove_user_sheet (GaleonEmbed *embed, gpointer sheet)
{
	GALEON_WRAPPER(wrapper);

	wrapper->RemoveOverrideStyleSheet ((nsIStyleSheet *) sheet);
}

extern "C" gboolean
mozilla_reload_proxy_autoconfiguration (const char* url)
{
	nsresult rv;

	nsCOMPtr<nsIProtocolProxyService> pps =
		do_GetService ("@mozilla.org/network/protocol-proxy-service;1",
			       &rv);
	if (NS_FAILED(rv) || !pps) return FALSE;

	rv = pps->ConfigureFromPAC (url);
	return rv ? TRUE : FALSE;
}

extern "C" GList*
mozilla_get_link_interface_items (GaleonEmbed *embed)
{
	GList *list = NULL;
	GALEON_WRAPPER_IF_FAIL (wrapper, NULL)
	
	wrapper->GetLinkInterfaceItems (&list);

	return list;
}

extern "C" gchar*
mozilla_make_security_tooltip (gpointer req)
{
	nsresult result;

	nsIRequest *request = (nsIRequest*)req;

	nsCOMPtr<nsIChannel> channel(do_QueryInterface(request, &result));
	if (NS_FAILED (result)) return NULL;

	nsCOMPtr<nsISupports> info;
	result = channel->GetSecurityInfo(getter_AddRefs(info));
	if (NS_FAILED (result) || !info) return NULL;

	nsCOMPtr<nsITransportSecurityInfo> secInfo(do_QueryInterface(info));
	if (!secInfo) return NULL;

	nsXPIDLString tooltip;
	result = secInfo->GetShortSecurityDescription(getter_Copies(tooltip));
	if (NS_FAILED (result) || !tooltip) return NULL;

	nsString string = nsString(tooltip);
	
	return convert_ns_string_to_c_string (string);
}

extern "C" gchar*
mozilla_security_level_string (guint state)
{
	gchar *level;

	switch (state)
	{
	case nsIWebProgressListener::STATE_IS_INSECURE:
		level = g_strdup (_("Insecure"));
		break;
	case nsIWebProgressListener::STATE_IS_BROKEN:
		level = g_strdup (_("Broken"));
		break;
	case nsIWebProgressListener::STATE_IS_SECURE|
	     nsIWebProgressListener::STATE_SECURE_HIGH:
		level = g_strdup (_("High"));
		break;
	case nsIWebProgressListener::STATE_IS_SECURE|
	     nsIWebProgressListener::STATE_SECURE_MED:
		level = g_strdup (_("Medium"));
		break;
	case nsIWebProgressListener::STATE_IS_SECURE|
	     nsIWebProgressListener::STATE_SECURE_LOW:
		level = g_strdup (_("Low"));
		break;
	default:
		level = g_strdup (_("Unknown"));
		break;
	}
	
	return level;
}

extern "C" gchar * 
mozilla_get_real_url (GaleonEmbed *embed)
{
	GALEON_WRAPPER_IF_FAIL (wrapper, NULL)

	nsCString url;
	nsresult rv = wrapper->GetRealURL (url);

	return (NS_SUCCEEDED (rv) && !url.IsEmpty()) ? g_strdup(url.get()) : NULL;
}

extern "C" void
mozilla_free (gpointer p)
{
	nsMemory::Free (p);
}

extern "C" void
mozilla_select_all (GaleonEmbed *embed)
{
	GALEON_WRAPPER (wrapper)

	wrapper->SelectAll ();
}

extern "C" void
mozilla_scroll_up (GaleonEmbed *embed)
{
	GALEON_WRAPPER (wrapper);

	wrapper->ScrollUp ();
}

extern "C" void
mozilla_scroll_down (GaleonEmbed *embed)
{
	GALEON_WRAPPER (wrapper);

	wrapper->ScrollDown ();
}

extern "C" void
mozilla_scroll_left (GaleonEmbed *embed)
{
	GALEON_WRAPPER (wrapper);

	wrapper->ScrollLeft ();
}

extern "C" void
mozilla_scroll_right (GaleonEmbed *embed)
{
	GALEON_WRAPPER (wrapper);

	wrapper->ScrollRight ();
}

extern "C" gchar *
mozilla_get_last_modified (GaleonEmbed *embed)
{
	GALEON_WRAPPER_IF_FAIL (wrapper, NULL);
	
	gchar *ret = NULL;
	wrapper->GetLastModified(&ret);
	return ret;
}

extern "C" GList *
mozilla_list_images (GaleonEmbed *embed)
{
	GALEON_WRAPPER_IF_FAIL (wrapper, NULL);

	GList *ret = NULL;
	wrapper->GetImages(&ret);
	return ret;
}

extern "C" GList *
mozilla_list_forms (GaleonEmbed *embed)
{
	GALEON_WRAPPER_IF_FAIL (wrapper, NULL);

	GList *ret = NULL;
	wrapper->GetForms(&ret);
	return ret;
}

extern "C" GList *
mozilla_list_links (GaleonEmbed *embed)
{
	GALEON_WRAPPER_IF_FAIL (wrapper, NULL);

	GList *ret = NULL;
	wrapper->GetLinks(&ret);
	return ret;
}

extern "C" GList *
mozilla_list_css (GaleonEmbed *embed)
{
	GList *csslist = NULL;
	nsresult result;
	GALEON_WRAPPER_IF_FAIL(wrapper, NULL)
	
	nsCOMPtr<nsIDOMStyleSheetList> list;
	result = wrapper->GetStyleSheets(getter_AddRefs(list));
	if (NS_FAILED(result) || !list) return NULL;
	
	PRUint32 count, i; 
	result = list->GetLength(&count);
	if (NS_FAILED(result)) return NULL;

	for (i = 0; i < count; i++)
	{
		nsCOMPtr<nsIDOMStyleSheet> item;
		result = list->Item(i, getter_AddRefs(item));
		if (NS_FAILED(result) || !item) continue;

		nsString string;
		result = item->GetHref(string);
		if (NS_FAILED(result)) continue;

		if (string.IsEmpty()) continue;

		LinkListItem *i = g_new0 (LinkListItem, 1);

		gchar *tmp = ToNewCString (string);
		i->url = g_strdup (tmp);
		nsMemory::Free (tmp);

		result = item->GetTitle(string);
		if (NS_SUCCEEDED(result))
		{
			gchar *tmp = ToNewCString (string);
			i->title = g_strdup (tmp);
			nsMemory::Free (tmp);
		}
		
		csslist = g_list_append (csslist, i);
	}

	return csslist;
}

extern "C" void
mozilla_evaluate_js (GaleonEmbed *embed, char *script)
{
	GALEON_WRAPPER(wrapper);

	wrapper->EvaluateJS (script);
}

extern "C" void
mozilla_register_js_listener (void)
{
	nsCOMPtr<nsIConsoleService> consoleService = 
		do_GetService (NS_CONSOLESERVICE_CONTRACTID);
	g_return_if_fail (consoleService != nsnull);

	JSConsoleListener *listener = new JSConsoleListener();
	consoleService->RegisterListener (listener);
}

extern "C" gchar *
mozilla_get_link_text (gpointer rn)
{
	nsresult result;
	int depth = 1;
	nsAutoString ltext;
	gchar *linktext = NULL;

	nsIDOMNode *rootnode = (nsIDOMNode *) rn;

	nsCOMPtr<nsIDOMNode> node;
	result = rootnode->GetFirstChild (getter_AddRefs(node));
	if (NS_FAILED(result)) return NULL;

	while (node && depth > 0)
	{
		nsString name;
		node->GetNodeName(name);
		if (name.EqualsWithConversion("#text", PR_TRUE))
		{
			nsCOMPtr<nsIDOMCharacterData> cdata = do_QueryInterface(node);
			nsString ctext;
			cdata->GetData(ctext);
			nsString s;
			s.AssignWithConversion(" ");
			ltext += s;
			ltext += ctext;
		}

		PRBool childs;
		node->HasChildNodes(&childs);

		if (childs)
		{
			nsCOMPtr<nsIDOMNode> childNode;
			node->GetFirstChild (getter_AddRefs(childNode));
			node = childNode;

			depth ++;
		}
		else
		{
			nsCOMPtr<nsIDOMNode> siblingNode;
			result = node->GetNextSibling (getter_AddRefs(siblingNode));
			if (NS_FAILED(result))
			{
				nsCOMPtr<nsIDOMNode> parentNode;
				result = node->GetParentNode (getter_AddRefs(parentNode));
				if (!NS_FAILED(result))
				{
					parentNode->GetNextSibling (getter_AddRefs(siblingNode));
				}
			}
			
			node = siblingNode;

		}
	}

	gchar *tmp = ToNewCString (ltext);
	gchar *clean = misc_string_strip_newline (g_strstrip (tmp));
	nsMemory::Free (tmp);

	if (strlen (clean) == 0)
	{
		g_free (clean);
		linktext = NULL;
	}
	else
		linktext = clean;

	return linktext;
}

extern "C" char *
mozilla_uri_get_parent (const char *uri)
{
	nsresult rv;

	nsCOMPtr<nsIURI> aURI;
	rv = NS_NewURI (getter_AddRefs(aURI), uri);
	if (NS_FAILED(rv) || !aURI) return NULL;

	nsCOMPtr<nsIURL> aURL = do_QueryInterface(aURI, &rv);
	if (NS_FAILED(rv) || !aURL) return NULL;

	nsCString dirPath;
	rv = aURL->GetDirectory (dirPath);
	if (NS_FAILED(rv) || !dirPath.Length()) return NULL;

	nsCString filePath;
	rv = aURL->GetFilePath (filePath);
	if (NS_FAILED(rv) || !filePath.Length()) return NULL;

	PRInt32 pathLength = filePath.Length();
	PRInt32 trailingSlash = filePath.RFind("/");

	if(pathLength < 2 || trailingSlash == -1)
	{
		return NULL;
	}

	if(trailingSlash != (pathLength-1))
	{
		aURI->SetPath(dirPath);
	}
	else
	{
		PRInt32 nextSlash = filePath.RFind("/",PR_FALSE,trailingSlash-1);
		nsCString parentPath;
		filePath.Left(parentPath, nextSlash);
		aURI->SetPath(parentPath);
	}

	nsCString spec;
	aURI->GetSpec(spec);

	return !spec.IsEmpty() ? g_strdup(spec.get()) : NULL;
}

extern "C" gboolean
mozilla_uri_has_parent (const char *uri)
{
	char *parentUri = mozilla_uri_get_parent (uri);
	if (!parentUri) return FALSE;
	else
	{
		g_free (parentUri);
		return TRUE;
	}
}

