/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "print.h"
#include "favicon.h"
#include "page_info.h"

#ifndef __mozilla_h
#define __mozilla_h

typedef struct
{
	/* image information */
	gchar *img; 
	gchar *imgalt;
	gchar *imglongdesc;
	gchar *imgtitle;
	gint imgwidth;
	gint imgheight;

	/* background image information */
	gchar *bgimg;

	/* link information */
	gchar *link; 
	gchar *linktext;
	gchar *linklang;
	gchar *linktarget;
	gchar *linktype;
	gchar *linkrel;
	gchar *linkrev;
	gchar *linktitle;

	/* smartbookmarks */
	gboolean is_smart;

	/* misc information */
	gboolean framed_page;
	gint context;
} WrapperContextInfo;

typedef struct
{
	WrapperContextInfo ctx;
	int key;
	gint modifier;
	gulong gdk_keyval;
} WrapperKeyEventInfo;

typedef struct
{
	WrapperContextInfo ctx;
	gint button;
	gint modifier;
	guint timestamp;
} WrapperMouseEventInfo;

typedef struct
{
	gchar *search_string;
	gboolean backwards;
	gboolean wrap;
	gboolean entire_word;
	gboolean match_case;
	gboolean search_frames;
} WrapperSearchProperties;

#define COOKIEPERMISSION 0
#define IMAGEPERMISSION 1

typedef enum
{
	ACTION_NONE,
	ACTION_VIEWSOURCE,
	ACTION_SETBACKGROUND,
	ACTION_SAVEFORHELPER,
	ACTION_FAVICON,
	ACTION_FAVICON_EDITOR,
	ACTION_SET_PIXMAP,
	ACTION_ADDBOOKMARK
} DownloadAction;

typedef struct
{
	gchar *url;
	gint width;
	gint height;
	gchar *alt;
	gchar *title;
} ImageListItem;

typedef struct
{
	gchar *action;
	gchar *method;
	gchar *name;
} FormListItem;

typedef struct
{
	gchar *url;
	gchar *title;
	gchar *rel;
} LinkListItem;

typedef struct
{
	gchar *url;
	gchar *selected_url;
	GtkWidget *pixmap;
} SetPixmapInfo;

typedef struct
{
	gchar *text;
	gchar *url;
	gchar *smarturl;
} SaveImageInfo;

#ifdef __cplusplus
extern "C"
{
#endif

/* Preferences */
gboolean mozilla_save_prefs (void);
gboolean mozilla_preference_set (const char *preference_name,
				 const char *new_value);
gboolean mozilla_preference_set_boolean (const char *preference_name,
					 gboolean  new_boolean_value);
gboolean mozilla_preference_set_int (const char *preference_name,
				     int new_int_value);
gchar *mozilla_preference_get (const char *preference_name);
gboolean mozilla_preference_get_boolean (const gchar *preference_name,
					 gboolean default_value);
gint mozilla_preference_get_int (const char *preference_name);
gboolean mozilla_preference_remove (const gchar *preference_name);

/* Wrapper initialization */
gpointer mozilla_wrapper_init (GaleonEmbed *embed);
gboolean mozilla_wrapper_destroy (GaleonEmbed *embed);
gboolean mozilla_pop_target_document (GaleonEmbed *embed);

/* Zoom */
gboolean mozilla_set_zoom (GaleonEmbed *embed, float f, gboolean reflow);
gboolean mozilla_get_zoom (GaleonEmbed *embed, float *f);

/* Persist */
gboolean mozilla_save_url (GaleonEmbed *embed, const char *url, 
			   char *filename, DownloadAction action,
			   gpointer *info);
gboolean mozilla_save_document (GaleonEmbed *embed, char *filename,
				const char *datapath, DownloadAction action,
				gboolean mainDoc);

/* Session History */
char *mozilla_session_history_get_url (GaleonEmbed *embed, int index);
char *mozilla_session_history_get_url_relative (GaleonEmbed *embed,
						int offset);
gboolean mozilla_session_history_get_all_titles (GaleonEmbed *embed,
						 char **titles[],
						 int *count, int *index);
gboolean mozilla_session_history_go (GaleonEmbed *embed, int index);
gboolean mozilla_copy_session_history (GaleonEmbed *embed,
				       GaleonEmbed *dest);

/* Clipboard */
gboolean mozilla_can_cut_selection (GaleonEmbed *embed);
gboolean mozilla_can_copy_selection (GaleonEmbed *embed);
gboolean mozilla_can_paste (GaleonEmbed *embed);
void mozilla_cut_selection (GaleonEmbed *embed);
void mozilla_copy_selection (GaleonEmbed *embed);
void mozilla_paste (GaleonEmbed *embed);

/* Utils */
gboolean mozilla_find (GaleonEmbed *embed,
		       WrapperSearchProperties *properties);
void mozilla_delete_temp_file_on_exit (const char *filename);
gboolean mozilla_reload (GaleonEmbed *embed);
gboolean mozilla_print (GaleonEmbed *embed, PrintInfo *info,
		    gboolean preview);
void mozilla_clear_cache (unsigned long cache);
gboolean mozilla_set_offline (GaleonEmbed *embed, gboolean offline);
void mozilla_free (gpointer p);
gchar *mozilla_get_main_document_url (GaleonEmbed *embed);
gchar *mozilla_get_document_url (GaleonEmbed *embed);
gboolean mozilla_reload_proxy_autoconfiguration (const char* url);
gboolean mozilla_get_favicon_location (GaleonEmbed *embed, gchar **url);
gboolean mozilla_show_java_console (void);
gboolean mozilla_show_javascript_console (void);
GList *mozilla_get_link_interface_items (GaleonEmbed *embed);
gchar *mozilla_get_document_url (GaleonEmbed *embed);
gchar *mozilla_get_document_title (GaleonEmbed *embed, gchar **title_utf8);
gchar *mozilla_get_frame_title (GaleonEmbed *embed);
gchar *mozilla_get_link_message (GaleonEmbed *embed,
				 gchar **link_message_utf8);
gchar *mozilla_get_js_status (GaleonEmbed *embed, gchar **js_status_utf8);
gchar *mozilla_get_real_url (GaleonEmbed *embed);
void mozilla_select_all (GaleonEmbed *embed);
void mozilla_register_js_listener (void);
void mozilla_evaluate_js (GaleonEmbed *embed, char *script);
char *mozilla_uri_get_parent (const char *uri);
gboolean mozilla_uri_has_parent (const char *uri);

/* Page information */
gchar *mozilla_get_last_modified (GaleonEmbed *embed);
GList *mozilla_list_images (GaleonEmbed *embed);
GList *mozilla_list_forms (GaleonEmbed *embed);
GList *mozilla_list_links (GaleonEmbed *embed);
GList *mozilla_list_css (GaleonEmbed *embed);
gchar *mozilla_get_link_text (gpointer rootnode);

/* Events */
gboolean mozilla_get_key_event_info (GaleonEmbed *embed,
				     gpointer event,
				     WrapperKeyEventInfo *info);
gboolean mozilla_get_mouse_event_info (GaleonEmbed *embed,
				       gpointer event,
				       WrapperMouseEventInfo *info);
void mozilla_free_context_info_sub (WrapperContextInfo *ctx);

/* Charsets and fonts*/
void mozilla_get_charsets (GHashTable **charsets,
			   GList **sorted_charset_titles);
void mozilla_force_character_set (GaleonEmbed *embed,
				  char *force_character_set);

/* Cookies manager */
void mozilla_set_url_permission (const gchar *imgURL, gint type,
				 gboolean allow);
GList *mozilla_list_cookies (void);
void mozilla_set_permission (GaleonEmbed *embed, gboolean permit, 
				    int type);
GList *mozilla_get_permissions (int type);
gboolean mozilla_remove_cookies (GList *gone);
gboolean mozilla_remove_permissions (GList *gone, int type);

/* Password Manager */
GList *mozilla_list_passwords (PasswordType type);
gboolean mozilla_remove_passwords (GList *gone, PasswordType type);

/* Stylesheets */
GList *mozilla_get_alternate_stylesheets (GaleonEmbed *embed);
void mozilla_set_alternate_stylesheet (GaleonEmbed *embed,
				       AlternateStyleSheet *css);
AlternateStyleSheet *mozilla_get_selected_stylesheet (GaleonEmbed *embed);
gpointer mozilla_set_user_sheet (GaleonEmbed *embed, gchar *sheet);
void mozilla_remove_user_sheet (GaleonEmbed *embed, gpointer sheet);

/* Security icon */
gchar *mozilla_make_security_tooltip (gpointer req);
gchar *mozilla_security_level_string (guint state);

/* Scrolling */
void mozilla_scroll_up (GaleonEmbed *embed);
void mozilla_scroll_down (GaleonEmbed *embed);
void mozilla_scroll_left (GaleonEmbed *embed);
void mozilla_scroll_right (GaleonEmbed *embed);

#ifdef __cplusplus
}
#endif 

#ifdef __cplusplus

#include <nscore.h>
#include "nsIDOMWindow.h"
#include "nsString.h"

extern "C"
{
gchar *mozilla_unicode_to_utf8 (const PRUnichar *uniStr);
gchar *mozilla_unicode_to_locale (const PRUnichar *uniStr);
PRUnichar *mozilla_utf8_to_unicode (const gchar *utfStr);
PRUnichar *mozilla_locale_to_unicode (const gchar *locStr);
GtkWidget *mozilla_find_gtk_parent (nsIDOMWindow *domWindow);
gchar *mozilla_unicode_to_page_locale (const PRUnichar *uniStr,
				       nsAutoString &platformCharset);
PRUnichar *mozilla_page_locale_to_unicode (const gchar *locStr,
					   nsAutoString &platformCharset);
}

#endif

#ifdef __cplusplus
extern "C"
{
#endif
gchar *mozilla_utf8_to_locale (const gchar* utfstr);
gchar *mozilla_locale_to_utf8 (const gchar* utfstr);
#ifdef __cplusplus
}
#endif 

#endif /* __mozilla_h */
