// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "organizer.h"
#include "about.h"
#include "collection.h"
#include "collectionview.h"
#include "constants.h"
#include "group.h"
#include "icons.h"
#include "mediasearch.h"
#include "mediaview.h"
#include "song.h"

#include <qaction.h>
#include <qapplication.h>
#include <qcursor.h>
#include <qdir.h>
#include <qdom.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistbox.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qsettings.h>
#include <qstatusbar.h>


static const char * whatsthis_image[] = {
    "16 16 3 1",
    "   c None",
    "o  c #000000",
    "a  c #000080",
    "o        aaaaa  ",
    "oo      aaa aaa ",
    "ooo    aaa   aaa",
    "oooo   aa     aa",
    "ooooo  aa     aa",
    "oooooo  a    aaa",
    "ooooooo     aaa ",
    "oooooooo   aaa  ",
    "ooooooooo aaa   ",
    "ooooo     aaa   ",
    "oo ooo          ",
    "o  ooo    aaa   ",
    "    ooo   aaa   ",
    "    ooo         ",
    "     ooo        ",
    "     ooo        "
};

static Organizer *INSTANCE = 0;


Organizer::Organizer(QWidget *parent)
    : QMainWindow(parent, "organizer"), destructing(false), playmode(Normal)
{
    if (INSTANCE) qFatal("Only one organizer allowed!");
    INSTANCE = this;

    setCaption(tr("MQ3 - Media Organizer"));

    // setup center widget
    QWidget *center = new QWidget(this);

    // mediaview in the center
    mediaview = new MediaView(center, "mediaview");
    connect(mediaview, SIGNAL(songSelected()), SIGNAL(songSelected()));

    // should the search have it's own dockwindow?
    mediasearch = new MediaSearch(center, "mediasearch");

    QVBoxLayout *v = new QVBoxLayout(center, 4, 4, "organizer center layout");
    v->addWidget(mediaview);
    v->addWidget(mediasearch);

    setCentralWidget(center);

    // collectionview in a dockwindow
    QDockWindow *dockwindow = new QDockWindow(QDockWindow::InDock, this);
    collview = new CollectionView(dockwindow, "collectionview");
    dockwindow->setCaption(tr("Collections View"));
    dockwindow->setResizeEnabled(true);
    dockwindow->setCloseMode(QDockWindow::Always);
    dockwindow->setWidget(collview);
    moveDockWindow( dockwindow, Left );

    // queueview in a dockwindow
    // dockwindow = new QDockWindow(QDockWindow::InDock, this);
    // QListView *queueview = new QListView(dockwindow);
    // dockwindow->setCaption(tr("Queue View"));
    // dockwindow->setResizeEnabled(true);
    // dockwindow->setCloseMode(QDockWindow::Always);
    // dockwindow->setWidget(queueview);
    // moveDockWindow( dockwindow, Bottom );

    setupActions();
    setupMenus();
    setupToolbars();
    statusBar();

    QSettings settings;

    // restore toplevel size
    int w, h;
    w = settings.readNumEntry("/MQ3/Organizer/width", 640);
    h = settings.readNumEntry("/MQ3/Organizer/height", 480);
    resize(w, h);

    // restore column widths
    for (h = 0; h < 4; h++) {
	w = settings.readNumEntry(QString("/MQ3/Organizer/MediaView/columnWidth/%1").
				  arg(h + 1), -1);
	if (w != -1)
	    mediaview->setColumnWidth(h, w);
    }

    QFile file(QDir::homeDirPath() + "/.mq3/toolbars");
    if (file.open(IO_ReadOnly)) {
	QTextStream ts( &file );
	ts >> *this;
	file.close();
    }
}

Organizer::~Organizer()
{
    destructing = true;
}

void Organizer::deinit()
{
    if (mediaview->isModified()) {
	if (autosave) {
	    mediaview->save();
	} else {
	    if (! QMessageBox::warning(0,
				       tr("Media Modified"),
				       tr("Do you want to save your Media changes?"),
				       tr("&Yes"),
				       tr("&No"),
				       QString::null, 0, 1))
		mediaview->save();
	}
    }

    if (collview->isModified()) {
	if (autosave) {
	    collview->save();
	} else {
	    if (! QMessageBox::warning(0,
				       tr("Collections Modified"),
				       tr("Do you want to save your Collection changes?"),
				       tr("&Yes"),
				       tr("&No"),
				       QString::null, 0, 1))
		collview->save();
	}
    }

    // save the toolbar position
    QFileInfo fileinfo(QDir::homeDirPath() + "/.mq3");
    if (! fileinfo.exists()) {
	QDir::home().mkdir(".mq3");
    }

    QFile file(QDir::homeDirPath() + "/.mq3/toolbars");
    if (file.open(IO_WriteOnly)) {
	QTextStream ts( &file );
	ts << *this;
	file.close();
    }

    // save toplevel size
    QSettings settings;
    settings.writeEntry("/MQ3/Organizer/width", width());
    settings.writeEntry("/MQ3/Organizer/height", height());

    // save media column widths
    settings.writeEntry("/MQ3/Organizer/MediaView/columnWidth/1",
			mediaview->columnWidth(0));
    settings.writeEntry("/MQ3/Organizer/MediaView/columnWidth/2",
			mediaview->columnWidth(1));
    settings.writeEntry("/MQ3/Organizer/MediaView/columnWidth/3",
			mediaview->columnWidth(2));
    settings.writeEntry("/MQ3/Organizer/MediaView/columnWidth/4",
			mediaview->columnWidth(3));
}

void Organizer::setupActions()
{
    actionSaveMedia = new QAction(tr("Save Media"),
				  tr("&Save Media"),
				  CTRL + Key_M,
				  this);
    actionSaveMedia->setStatusTip(tr("Save changes to your media."));
    connect(actionSaveMedia, SIGNAL(activated()), mediaview, SLOT(save()));

    actionSaveColl = new QAction(tr("Save Collections"),
				 tr("Save &Collections"),
				 CTRL + Key_S,
				 this);
    actionSaveColl->setStatusTip(tr("Save changes to your collections."));
    connect(actionSaveColl, SIGNAL(activated()), collview, SLOT(save()));

    actionAbout = new QAction(tr("Display information about MQ3"),
			      tr("&About..."),
			      0,
			      this);
    actionAbout->
	setWhatsThis(tr("<b>Display information about MQ3.</b>"
			"<p>A dialog will open showing the version and "
			"contact information for the MQ3 Media Player.</p>"));
    connect(actionAbout, SIGNAL(activated()), this, SLOT(about()));

    actionAboutQt = new QAction(tr("Display information about the Qt toolkit"),
				tr("About &Qt..."),
				0,
				this);
    actionAboutQt->
	setWhatsThis(tr("<b>Display information about the Qt toolkit.</b>"
			"<p>A dialog will open showing the version and "
			"contact information for the Qt toolkit used to "
			"build this program.</p>"));
    connect(actionAboutQt, SIGNAL(activated()), this, SLOT(aboutQt()));

    actionWhatsThis = new QAction(tr("What's this?"),
				  QIconSet(whatsthis_image, whatsthis_image),
				  tr("&What's this?"),
				  SHIFT + Key_F1,
				  this);
    actionWhatsThis->
	setStatusTip(tr("\"What's this?\" context sensitive help."));
    actionWhatsThis->
	setWhatsThis(tr("<b>That's me!</b>"
			"<p>In What's This?-Mode, the mouse cursor shows "
			"an arrow with a questionmark, and you can click "
			"on the interface elements to get a short "
			"description of what they do and how to use them. "
			"In dialogs, this feature can be accessed using "
			"the context help button in the titlebar.</p>"));
    connect(actionWhatsThis, SIGNAL(activated()), this, SLOT(whatsThis()));

    actionClose = new QAction(tr("Close"),
			      tr("&Close"),
			      CTRL + Key_W,
			      this);
    actionClose->setStatusTip(tr("Close the Organizer."));
    connect(actionClose, SIGNAL(activated()), this, SLOT(close()));

    actionPrevious = new QAction(tr("Previous"),
				 tr("Previous"),
				 0,
				 this);
    actionPrevious->setStatusTip(tr("Play previous song"));
    actionPrevious->setWhatsThis(tr("Play previous song"));
    connect(actionPrevious, SIGNAL(activated()), this, SIGNAL(Previous()));

    actionPlay = new QAction(tr("Play"),
			     tr("Play"),
			     0,
			     this);
    actionPlay->setStatusTip(tr("Play selected song"));
    actionPlay->setWhatsThis(tr("Play selected song"));
    connect(actionPlay, SIGNAL(activated()), this, SIGNAL(Play()));

    actionPause = new QAction(tr("Pause"),
			      tr("Pause"),
			      0,
			      this);
    actionPause->setStatusTip(tr("Pause current song"));
    actionPause->setWhatsThis(tr("Pause current song"));
    connect(actionPause, SIGNAL(activated()), this, SIGNAL(Pause()));

    actionStop = new QAction(tr("Stop"),
			     tr("Stop"),
			     0,
			     this);
    actionStop->setStatusTip(tr("Stop current song"));
    actionStop->setWhatsThis(tr("Stop current song"));
    connect(actionStop, SIGNAL(activated()), this, SIGNAL(Stop()));

    actionNext = new QAction(tr("Next"),
			     tr("Next"),
			     0,
			     this);
    actionNext->setStatusTip(tr("Play next song"));
    actionNext->setWhatsThis(tr("Play next song"));
    connect(actionNext, SIGNAL(activated()), this, SIGNAL(Next()));

    mediaview->setupActions();
    collview->setupActions();
}

void Organizer::setupMenus()
{
    mediaview->setupMenus();
    collview->setupMenus();

    QPopupMenu *filemenu, *helpmenu;
    filemenu = new QPopupMenu(this);
    helpmenu = new QPopupMenu(this);

    // filemenu
    actionSaveMedia->addTo(filemenu);
    actionSaveColl->addTo(filemenu);
    filemenu->insertSeparator();
    actionClose->addTo(filemenu);

    // help menu
    actionAbout->addTo(helpmenu);
    actionAboutQt->addTo(helpmenu);
    helpmenu->insertSeparator();
    actionWhatsThis->addTo(helpmenu);

    menuBar()->insertItem(tr("&File"), filemenu);
    menuBar()->insertItem(tr("&Media"), mediaview->mainMenu());
    menuBar()->insertItem(tr("&Collections"), collview->mainMenu());
    menuBar()->insertSeparator();
    menuBar()->insertItem(tr("&Help"), helpmenu);
}

void Organizer::setupToolbars()
{
    mediaview->setupToolbars();
    collview->setupToolbars();

    QToolBar *ctoolbar, *wtoolbar;
    ctoolbar = new QToolBar(tr("Controls"), this);
    wtoolbar = new QToolBar(tr("What's This?"), this);

    // controls
    actionPrevious->addTo(ctoolbar);
    actionPause->addTo(ctoolbar);
    actionPlay->addTo(ctoolbar);
    actionStop->addTo(ctoolbar);
    actionNext ->addTo(ctoolbar);

    // what's this?
    actionWhatsThis->addTo(wtoolbar);
}

void Organizer::configChanged(QSettings &settings)
{
    QString val = settings.readEntry("/MQ3/Organizer/playMode");
    playmode = Normal;
    if (val == "normal")
	playmode = (Organizer::Normal);
    else if (val == "random")
	playmode = (Organizer::Random);
    else if (val == "onesong")
	playmode = (Organizer::OneSong);
    else if (val == "repeatall")
	playmode = (Organizer::RepeatAll);
    else if (val == "repeatone")
	playmode = (Organizer::RepeatOne);

    setUsesBigPixmaps(settings.readBoolEntry("/MQ3/Organizer/useLargeIcons", false));
    setUsesTextLabel(settings.readBoolEntry("/MQ3/Organizer/useTextLabels", false));

    mediasearch->setUseRegexp(settings.readBoolEntry("/MQ3/Organizer/regexpSearching",
						     false));

    autosave = settings.readBoolEntry("/MQ3/Organizer/saveChangesOnExit", false);

    mediaview->loadIcons();
    collview->loadIcons();

    QIconSet iconset = IconLoader::load("organizer");
    setIcon(iconset.pixmap(QIconSet::Large, QIconSet::Normal));
    actionSaveMedia->setIconSet(IconLoader::load("savemedia"));
    actionSaveColl->setIconSet(IconLoader::load("savecollections"));
    actionClose->setIconSet(IconLoader::load("close"));
    actionPrevious->setIconSet(IconLoader::load("prev"));
    actionPlay->setIconSet(IconLoader::load("play"));
    actionPause->setIconSet(IconLoader::load("pause"));
    actionStop->setIconSet(IconLoader::load("stop"));
    actionNext->setIconSet(IconLoader::load("next"));
}

void Organizer::load()
{
    mediaview->load();
    collview->load();
}

void Organizer::about()
{
    AboutDialog d(this);
    d.setCaption(d.caption().arg(VERSION));
    d.titleLabel->setText(d.titleLabel->text().arg(VERSION));
    d.thanksListBox->insertItem("Erik Bakke");
    d.thanksListBox->insertItem("Marius Nita");
    d.exec();
}

void Organizer::aboutQt()
{
    QMessageBox::aboutQt(this, tr("MQ3 Media Player"));
}

Organizer *Organizer::instance()
{
    return INSTANCE;
}
