/*
 * videoio.cxx
 *
 * Video I/O Device handers
 *
 * Open H323 Library
 *
 * Copyright (c) 1999-2000 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: videoio.cxx,v $
 * Revision 1.11  2001/09/21 03:22:47  robertj
 * Removed unused function
 *
 * Revision 1.10  2001/09/20 04:13:17  dereks
 * Implement faster H323VideoDevice::Redraw routine from Morganti Tiziano.
 * Thanks for your help.
 *
 * Revision 1.9  2001/02/09 13:31:22  robertj
 * Fixed conflict with #pragma interface in pwlib
 *
 * Revision 1.8  2001/02/09 05:35:02  craigs
 * Fixed typo in pragma
 *
 * Revision 1.7  2001/02/09 05:13:56  craigs
 * Added pragma implementation to (hopefully) reduce the executable image size
 * under Linux
 *
 * Revision 1.6  2000/12/19 22:33:44  dereks
 * Adjust so that the video channel is used for reading/writing raw video
 * data, which better modularizes the video codec.
 *
 * Revision 1.5  2000/05/02 04:32:28  robertj
 * Fixed copyright notice comment.
 *
 * Revision 1.4  2000/02/04 05:11:19  craigs
 * Updated for new Makefiles and for new video transmission code
 *
 * Revision 1.3  1999/11/01 00:52:00  robertj
 * Fixed various problems in video, especially ability to pass error return value.
 *
 * Revision 1.2  1999/09/21 14:10:18  robertj
 * Windows MSVC compatibility.
 *
 * Revision 1.1  1999/09/21 08:10:42  craigs
 * Created initial revision of video support
 *
 */

#include <ptlib.h>


#ifdef __GNUC__
#pragma implementation "videoio.h"
#endif

#include <videoio.h>


/////////////////////////////////////////////////////////////////////////////

H323VideoDevice::H323VideoDevice() 
{
  SetFrameSize(0,0);
  rgbReverseOrder = FALSE;
}


BOOL H323VideoDevice::Close()
{
  return TRUE;
}


#define LIMIT(x) (BYTE)(((x>0xffffff)?0xff0000:((x<=0xffff)?0:x&0xff0000))>>16)


// By removeing the yuvtorgb() calls, this routine is 50% faster
BOOL H323VideoDevice::Redraw(const void * frame)
{
  unsigned size, width, height;
  width = height = 0; 
  GetFrameSize(width,height);
  size = width * height;

  // get pointers to the data
  const BYTE * yplane  = (BYTE *)frame;
  const BYTE * uplane  = yplane + size;
  const BYTE * vplane  = yplane + size + (size >> 2);

  // get temporary buffers
  PBYTEArray rgbLine(width*3);  // draw 2 lines at once
  PBYTEArray rgbLine2(width*3); // draw 2 lines at once

  unsigned x, y;
  for (y = 0; y < height; y+=2) {

    const BYTE * yline  = yplane + (y * width);
    const BYTE * yline2 = yline  + width;
    const BYTE * uline  = uplane + ((y >> 1) * (width >> 1));
    const BYTE * vline  = vplane + ((y >> 1) * (width >> 1));
        
    BYTE * rgb  = rgbLine.GetPointer();
    BYTE * rgb2 = rgbLine2.GetPointer();

    for (x = 0; x < width; x+=2) {
          long Cr = *uline++ - 128;     // calculate once for 4 pixels
          long Cb = *vline++ - 128;
          long lrc = 104635 * Cb;
          long lgc = -25690 * Cr + -53294 * Cb;
          long lbc = 132278 * Cr;

          if (rgbReverseOrder)  
          {
                  long tmp;     // exchange red component and blue component
                  tmp=lrc;
                  lrc=lbc;
                  lbc=tmp;
          }
          
          long Y  = *yline++ - 16;      // calculate for every pixel
	  if (Y < 0)
	    Y = 0;
          long l  = 76310 * Y;
          long lr = l + lrc;
          long lg = l + lgc;
          long lb = l + lbc;

	  *rgb++ = LIMIT(lr);
	  *rgb++ = LIMIT(lg);
	  *rgb++ = LIMIT(lb);         
          
	  Y  = *yline++ - 16;       // calculate for every pixel
	  if (Y < 0)
	    Y = 0;
          l  = 76310 * Y;
          lr = l + lrc;
          lg = l + lgc;
          lb = l + lbc;

          *rgb++ = LIMIT(lr);
	  *rgb++ = LIMIT(lg);
	  *rgb++ = LIMIT(lb);         
          
          Y  = *yline2++ - 16;     // calculate for every pixel
	  if (Y < 0)
	    Y = 0;
          l  = 76310 * Y;
          lr = l + lrc;
          lg = l + lgc;
          lb = l + lbc;

	  *rgb2++ = LIMIT(lr);
	  *rgb2++ = LIMIT(lg);
	  *rgb2++ = LIMIT(lb);        
          
	  Y  = *yline2++ - 16;      // calculate for every pixel
	  if (Y < 0)
	    Y = 0;
          l  = 76310 * Y;
          lr = l + lrc;
          lg = l + lgc;
          lb = l + lbc;

          *rgb2++ = LIMIT(lr);
	  *rgb2++ = LIMIT(lg);
	  *rgb2++ = LIMIT(lb);        
    }

    if (!WriteLineSegment(0, y, width, rgbLine))
      return FALSE;
    if (!WriteLineSegment(0, y+1, width, rgbLine2))
      return FALSE;
  }

  return TRUE;
}


/////////////////////////////////////////////////////////////////////////////

PPMVideoOutputDevice::PPMVideoOutputDevice()
{
  frameNumber = 0;
  deviceID=00;//Used to distinguish between input and output device.
}


PPMVideoOutputDevice::PPMVideoOutputDevice(int idno)
{
 frameNumber = 0;
  deviceID=idno;//Used to distinguish between input and output device.
}

BOOL PPMVideoOutputDevice::Redraw(const void * frame)
{
  unsigned  width,height;
  width = height = 0;
  GetFrameSize(width,height);

  PFilePath fn = psprintf("%02dframe%i.ppm",deviceID, frameNumber++);
  if (!file.Open(fn, PFile::WriteOnly))
    return FALSE;

  file << "P6 " << width  << " " << height << " " << 255 << "\n";

  buffer.SetSize(width * height * 3);

  H323VideoDevice::Redraw(frame);

  if (!file.Write(buffer, width*height*3))
    return FALSE;

  buffer.SetSize(0);

  return file.Close();
}


BOOL PPMVideoOutputDevice::WriteLineSegment(int x, int y, unsigned len, const BYTE * data)
{
  unsigned width,height;
  width = height = 0;
  GetFrameSize(width,height);

  PINDEX offs = 3 * (x + (y * width));
  memcpy(buffer.GetPointer() + offs, data, len*3);
  return TRUE;
}


// End of File ///////////////////////////////////////////////////////////////
